(in-ns 'buckshot.backend)
(require '[taoensso.carmine :as redis])

(defmacro with-conn [& body]
  `(redis/with-conn ~'pool ~'spec
     ~@body))

(defn- success? [results]
  (and (seq results)
       (every? #{1} results)))

(defrecord RedisBackend [pool spec]
  IBackend
  (get-colls [_]
    (->> (with-conn
           (redis/keys "*"))
         (map keyword)))
  (get-all [_ coll]
    (with-conn
      (redis/zrange (name coll) 0 -1)))
  (get-min [_ coll]
    (-> (with-conn
          (redis/zrangebyscore (name coll) "-inf" "+inf" "LIMIT" 0 1))
        first))
  (has-score? [_ coll score]
    (-> (with-conn
          (redis/zrangebyscore (name coll) score score "LIMIT" 0 1))
        seq))
  (has-item? [_ coll item]
    (with-conn
      (redis/zscore (name coll) item)))
  (atomically [_ f]
    (-> (with-conn
          (redis/atomically [] (f)))
        success?))
  (add! [_ coll score item]
    (redis/zadd (name coll) score item))
  (remove! [_ coll item]
    (redis/zrem (name coll) item))
  (delete! [_ coll]
    (redis/del (name coll)))
  (publish! [_ channel message]
    (-> (with-conn
          (redis/publish (name channel) message))
        pos?))
  (subscribe! [_ channel f]
    (redis/with-new-pubsub-listener spec
      {(name channel) (fn [[type channel message]]
                        (when (= type "message")
                          (f message)))}
      (redis/subscribe (name channel)))))

(defn- make-redis [{:keys [pool spec] :as params}]
  (RedisBackend. (or pool (redis/make-conn-pool))
                 (merge (redis/make-conn-spec) spec)))
