(ns borg.transport.http
  (:require [borg.internal.lang :refer [-?>>]]
            [borg.internal.util :as util]
            [borg.transport.interface :refer :all]
            [cheshire.core :as json]
            [clj-http.core :as http]
            [clj-http.client :as http-client]
            [clj-http.conn-mgr :as conn]
            [clj-http.cookies :as cookie]
            [clojure.set :as set]
            [clojure.tools.logging :as lg]
            [ring.adapter.jetty :as jetty]
            [ring.middleware.session :as s])
  (:import java.util.Date
           org.eclipse.jetty.server.ssl.SslSelectChannelConnector
           org.apache.http.conn.ssl.SSLSocketFactory
           org.apache.http.conn.scheme.Scheme))

(defn ssl? [server]
  (not= nil (:ssl-opt server)))

(defn ssl-client
  "Renames the keystore config keys for clj-http."
  [server]
  (set/rename-keys (:ssl-opt server)
                   {:keystore :trust-store
                    :password :trust-store-pass}))

(defn ssl-server
  "Renames the keystore config keys for ring."
  [server]
  (set/rename-keys (:ssl-opt server)
                   {:password :key-password}))

(defn make-request
  [server handler-spec]
  (merge (ssl-client server)
         {:body (json/generate-string handler-spec)
          :content-type :json
          :socket-timeout 30000
          :conn-timeout 30000
          :accept :json
          :cookie-store (:cookie-store server)}))

(defn ssl-req
  "For performing a request over ssl. Removes the host
   name verifier for certificates."
  [url params]
  (let [c (conn/make-regular-conn-manager params)]
    (-> (.getSchemeRegistry c)
        (.getScheme "https")
        (.getSchemeSocketFactory)
        (.setHostnameVerifier SSLSocketFactory/ALLOW_ALL_HOSTNAME_VERIFIER))
    (->> (assoc params :connection-manager c)
         (http-client/post url))))

(defn pre-session
  "Checks the time of the last request for this session,
   if it was longer ago than 'timeout' then remove the
   :authenticated and :user keys."
  [timeout session]
  (let [now (-> (Date.) .getTime)]
    (if (-?>> (:last-request session)
              (.getTime)
              (- now)
              (> timeout))
      session
      (dissoc session :authenticated :user))))

(defn post-session
  "If response has :authenticated and it is true, set the session
   keys :authenticated and :user. If :authenticated is false remove
   those keys from the session."
  [session response]
  (-> (if (contains? response :authenticated)
        (if (:authenticated response)
          (assoc session
            :authenticated true
            :user (:user response))
          (dissoc session :authenticated :user))
        session)
      (assoc :last-request (Date.))))

(defn request-handler [timeout handle-exec]
  (s/wrap-session
    (fn [req]
      (let [session (pre-session timeout (:session req))
            result (-> (:body req)
                       (slurp)
                       (json/parse-string true)
                       (handle-exec (:user session)
                                    (:authenticated session)))]
        {:body (-> (dissoc result :user :authenticated) json/generate-string)
         :headers {"Content-Type" "text/json"}
         :status 200
         :session (post-session session result)}))))

(defn configurator
  "If using ssl remove all non ssl listeners.
   Set a graceful shutdown period of 1 second."
  [ssl? server]
  (when ssl?
    (doseq [c (.getConnectors server)]
      (println (type c))
      (when-not (or (nil? c) (instance? SslSelectChannelConnector c))
        (.removeConnector server c))))
  (.setGracefulShutdown server 1000))

(defrecord HTTP [cookie-store session-timeout ssl-opt]
  ITransport
  (start-server [_ exec-fn port]
    (let [ssl (ssl? _)
          server
          (->> (when ssl {:ssl? true :ssl-port port})
               (merge (ssl-server _)
                      {:configurator #(configurator ssl %)
                       :port port
                       :join? false})
               (jetty/run-jetty (request-handler (:session-timeout _) exec-fn)))]
      (lg/info "Listening on port" port)
      server))

  (stop-server [_ server]
    (lg/info "stoping" server)
    (.stop server)
    (lg/info "Server shutting down"))

  (create-client [_ host port]
    (str "http://" host ":" port))

  (close-client [_ client]
    nil)

  (send-command [_ client handler-spec]
    (-> ((if (ssl? _)
           ssl-req
           http-client/post)
         client (make-request _ handler-spec))
        (:body)
        (json/parse-string true))))

(defn init
  "If using ssl takes the keys
  :keystore - the path to the jks keystore
  :password - password for the keystore.

  The default session timeout before requiring to
  reauthenticate is 5 minutes, to set a custom timeout
  supply the key :timeout in minutes."
  [{:keys [keystore password timeout] :as opt}]
  (->> (when (or keystore password)
         (dissoc opt :timeout))
       (HTTP. (cookie/cookie-store)
              (-> (or timeout 5)
                  (util/unit->ms :minute)))))
