;; Owner: marshall@readyforzero.com
;;
;; Checks out source code from a git repo, and manages existing
;; revisions of the repo using a releases directory and a symlink to
;; the current release.
;;
;; code-dir /
;;   releases /
;;     <commit-hash-1>
;;     <commit-hash-2>
;;   current -> releases/<commit-hash-2>
;;
;; If a commit already exists in the releases directory it won't check
;; it out again, but that behavior can be overridden.

(ns borg.state.types.repo
  (:require [borg.borglet.util.io :as borg-io]
            [borg.state.graph :as g]
            [borg.state.types.core :refer [defop sh-result node-path path try-to chain-op]]
            [borg.state.types.fs :as fs]
            [borg.state.types.provided :as provided]
            [clojure.core.match :as m]
            [clojure.java.shell :as sh]
            [clojure.string :as string]
            [me.raynes.fs :as me-fs]))

(defn local-provides [base addition]
  (-> (name base)
      (str "-" addition)
      (keyword)))

(defn make-releases-dir [provides path user app-dir-node]
  (provided/directory (str path "/releases")
               {:user user
                :permissions "0755"
                :provides (local-provides provides "rev-dir")
                :requires [app-dir-node]}))

(defn make-app-dir [provides path user]
  (provided/directory path
               {:user user
                :permissions "0755"
                :provides (local-provides provides "app-dir")}))

(defn make-directories [provides path user]
  (->> (make-app-dir provides path user)
       (make-releases-dir provides path user)
       g/simulate-send
       g/check-nodes
       (mapcat vals)
       (mapcat :actions)))

(defop deploy-repo [repo commit user ssh-key revisions-dir]
  (let [args {:repo repo :commit commit :user user :ssh-key ssh-key :revisions-dir revisions-dir}]
    (try-to (str "deploy revision " args)
            (borg-io/git-deploy-revision args))))

(defn check-release
  "Expected attributes in :attr
   :commit - the commit hash
   :code-dir - the directory that will contain the releases.
               directory and the 'current' symlink.
   :repo - url to the repo.
   :user - the user who should own the files and directories.
   :ssh-key - optional path to an ssh key used to checkout the repo."
  [node]
  (let [provides (:provides node)
        {commit :commit dir :code-dir user :user} (:attrs node)
        releases-dir (str dir "/releases")
        release-dir (str releases-dir "/" commit)
        symlink  (str dir "/current")
        repo-args (-> (select-keys (:attrs node) [:repo :commit :user :ssh-key])
                      (assoc :revisions-dir releases-dir))
        dir-actions (make-directories provides dir user)
        exists? (me-fs/exists? release-dir)]
    ;; !! The logic below is not self-evident.
    ((node-path
      ~@dir-actions
      (and (-> node :attrs :force) exists?) => (chain-op (fs/delete release-dir)
                                                         (node-deploy-repo node))
      (not exists?) => deploy-repo
      (not (fs/link? release-dir symlink)) => (fs/link release-dir symlink))
     node)))

(defmethod g/from-wire "repo" [node]
  (let [code-dir (g/node-attr node :code-dir)]
    (assoc-in node [:attrs :revisions-dir] (str code-dir "/releases"))))

(defmethod g/check-node "repo"
  [node _ _]
  (check-release node))
