;; Owner: wolfson@readyforzero.com
;; Predefined node types
(ns borg.state.types.provided
  (:require [borg.state.util :as u]))

;; validators for attributes of a node. E.g.,
;; (user u/string-or-keyword? name)
;; requires that the :user attr be passed as a string or a keyword and
;; converts it to a string.
;; Can be called multiple times to define additional validators.
;; If no validator is defined for an attribute, it is assumed to be valid.
(u/make-validators (user u/string-or-keyword? name)
                   (group u/string-or-keyword? name)
                   (permissions string?)
                   (template #(and (map? %)
                                   (let [k (keys %)]
                                     (and (== 1 (count k))
                                          (#{:file :string :literal :s3} (first k)))))
                             #(do {:type (first (keys %))
                                   :val (first (vals %))}))
                   (env map?)
                   (block? (some-fn true? false? nil?))
                   (args #(every? (fn [x] (or (string? x)
                                             (fn? x))) %))
                   (pkgname string?)
                   (version string?)
                   (gid integer?)
                   (delete? true?)
                   (on-child-ops #(and (map? %)
                                       (every? (comp (partial == 1) count) (vals %))
                                       (every? #{:sighup :restart :shutdown-first} (mapcat vals (vals %)))))
                   (groups #(and (sequential? %)
                                 (every? u/string-or-keyword? %))))


(u/node-type file :arguments [pathname] :required-attrs [user permissions template]
             :optional-attrs [context-fn group]
             :produces-ops [delete create chown chmod write])

(u/node-type directory :arguments [pathname] :required-attrs [user permissions]
             :optional-attrs [group]
             :produces-ops [delete create-dir chown chmod])

;; for changing the permissions/ownership of a file/directory without
;; bothering to inspect its contents.
;; will report an error if the file/dir doesn't exist.
(u/node-type file-permissions :arguments [pathname] :required-attrs [permissions user]
             :optional-attrs [group]
             :produces-ops [chown chmod])

(u/node-type process :arguments [pathname] :required-attrs [user group env args exec-dir]
             :optional-attrs [block? on-child-ops]
             :produces-ops [run-command delete create create-dir chown chmod write link ensure-running
                            service-down sighup])

(u/node-type package :arguments [pkgname version]
             :produces-ops [install])

(u/node-type group :arguments [name] :optional-attrs [gid]
             :produces-ops [create-group mod-group])

;; every user has its own group. the optional "groups" is for
;; additional groups to which the user belongs.
(u/node-type user :arguments [name] :optional-attrs [groups delete?]
             :produces-ops [create-user delete-user shred-user-home mod-user-groups])

(u/node-type repo
             :arguments [commit]
             :required-attrs [user code-dir repo user]
             :optional-attrs [ssh-key]
             :produces-ops [deploy-repo delete link
                            create-dir chown chmod])

(u/node-type drive
             :arguments [drive]
             :required-attrs [partitions]
             :produces-ops [unmount partition-drive])
