(ns borg.state.types.internal.core
  (:require [clojure.string :as str]
            [macroparser.bindings :as b]
            [macroparser.parsers :as p]
            [the.parsatron :as parsatron]))

(defn replacement-bindings [node-sym names]
  (parsatron/run (b/binding-form) [[{{:keys (vec names)} :attrs :as node-sym}]]))

(defn update-body-op [op-name arg-names body]
  (let [uncensored-arg-names (remove #(-> % meta :censor) arg-names)]
    (list
     {:op (keyword op-name)
      :args `(zipmap ~(mapv keyword uncensored-arg-names) ~(vec uncensored-arg-names))
      :body (str `(fn [] ~@body))
      :fn `(fn [] ~@body)})))

(defn update-arity [updater op-name arity]
  (let [body (:body arity)
        bindings (-> arity :params b/bound-symbols)]
    (assoc arity :body (updater op-name bindings body))))

(defn prefix-node-sym [s]
  (if (and (symbol? s)
           (not= '=> s))
    (let [str-s (str s)
          [before after] (str/split str-s #"/" 2)
          str-s (or after before)
          prefix (and after before)]
      (if (.startsWith str-s "node-")
        s
        (symbol prefix (str "node-" str-s))))
    s))

(parsatron/defparser plain-path []
  (p/lift #(do {:type :plain :pred %}) (parsatron/either
                                        (p/seq)
                                        (p/symbols-but '=>))))

(parsatron/defparser explicit-path []
  (parsatron/let->> [pred (parsatron/either (p/seq) (p/anything-but '=>))
                     _ (p/symbol '=>)
                     op (parsatron/either (p/seq) (p/symbols-but '=>))]
                    (parsatron/always {:pred pred
                                       :op op
                                       :type :explicit})))

(parsatron/defparser parse-path-elems []
  (p/>>1 (parsatron/many (p/choice+ (explicit-path) (plain-path))) (parsatron/eof)))

(defn trans-splices [elem]
  (if (and (= :plain (:type elem))
           (seq? (:pred elem))
           (= 'clojure.core/unquote-splicing (first (:pred elem))))
    {:type :splice :lst (second (:pred elem))}
    elem))

(defn unparse-op [op argsym]
  (if (symbol? op)
    (list op argsym)
    op))

(defn unparse-plain [op]
  (if (symbol? op)
    `(fn [arg#] [(~op arg#)])
    `(fn [arg#] [~op])))

(defn un-pred [s]
  (symbol (apply str (butlast (name s)))))

(defn is-pred? [s]
  (and (symbol? s)
       (= \? (last (name s)))))
