;; Owner: wolfson@readyforzero.com
;; Defines actions and state node functions for "file", "directory",
;; and "file-permissions".
;; types (see fs-node-checker at the bottom of the file).
;; Some of these functions are also used by other node implementations that
;; need to manipulate or access the filesystem.
(ns borg.state.types.fs
  (:require [aws.sdk.s3 :as s3]
            [borg.aws.core :as aws]
            [borg.state.graph :as g]
            [borg.state.util :as u]
            [clojure.core.match :as m]
            [clojure.java.shell :as sh]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [digest]
            [me.raynes.fs :as fs]
            [stencil.core :as stencil])
  (:use [borg.state.types.core :only [defop defnodefn try-to sh-result node-path]]
        [borg.state.util :only [defnmatch]])
  (:import [java.io ByteArrayOutputStream]))

(defn stat [pathname]
  (let [{:keys [exit out err]} (sh/sh "stat" "-c" "%u %U %g %G %a %F" pathname)]
    (when (== exit 0)
      (let [[uid uname gid gname perms ftype] (str/split (first (str/split-lines out)) #" " 6)]
        {:uid uid
         :uname uname
         :gid gid
         :gname gname
         :perms perms
         :ftype (case ftype
                  "regular file" "file"
                  "symbolic link" "link"
                  "directory" "directory"
                  "unknown")}))))

(defn file-type [pathname]
  (:ftype (stat pathname)))

(defnodefn link? [target link]
  (-> (sh/sh "readlink" "-f" link)
      :out
      (str/split #"\n")
      first
      (= target)))

;; the fs lib sometimes returns false on failure, and sometimes throws
;; an exception. Normalize to always throw an exception, w/ a supplied
;; message.
(defn wrap-attempt [msg success?]
  (if success?
    (u/ok)
    (throw (Exception. msg))))

(defn- delete-thing [deletion-fn pathname]
  (try-to (str "delete " pathname)
          (wrap-attempt (str "Could not delete " pathname)
                        (deletion-fn pathname))))

(defop link [target link]
  (u/chain-status
   (if (fs/exists? link)
     (delete-thing fs/delete link)
     (u/ok))
   (sh-result (sh/sh "ln" "--symbolic" target link))))

(defop delete-file [pathname]
  (delete-thing fs/delete pathname))

(defop delete [pathname]
  (case (file-type pathname)
    "file" (delete-thing fs/delete pathname)
    "link" (delete-thing fs/delete pathname)
    "directory" (delete-thing fs/delete-dir pathname)
    "unknown" (u/error (str "Don't know how to delete " pathname))
    nil (u/error (str pathname " does not exist, cannot delete"))))

;; fs chmod requires symbolic modes and just has a boolean flag for
;; user/not-user, no specification of group/other
(defop chmod [pathname permissions]
  (sh-result (sh/sh "chmod" permissions pathname)))

(defnodefn chmod? [permissions stat]
  (if (and (== 4 (count permissions))
           (= \0 (first permissions)))
    (not= (apply str (rest permissions)) (:perms stat))
    (not= permissions (:perms stat))))

;; fs has no chown :(
(defop chown [pathname user group]
  (sh-result
   (if group
     (sh/sh "chown" (str user ":" group) pathname)
     (sh/sh "chown" user pathname))))

(defnodefn chown? [user group stat]
  (or (not= user (:uname stat))
      (and group (not= group (:gname stat)))))

(defop create [pathname]
  (try-to (str "create " pathname)
          (wrap-attempt (str "Could not create " pathname)
                        (fs/create (io/as-file pathname)))))

(defop create-dir [pathname]
  (try-to (str "create " pathname)
          (wrap-attempt (str "Could not create " pathname)
                        (fs/mkdir pathname))))

(defop write [pathname ^:censor contents]
  (try-to (str "write to " pathname)
          (with-open [o (io/output-stream pathname)]
            (.write o contents))))

(defnodefn write? [pathname contents]
  (or (not (fs/exists? pathname))
      (not= (digest/md5 contents) (digest/md5 (io/file pathname)))))

(defnmatch render-template
  ([{:type :string :val t} context] {:literal (stencil/render-string t context)})
  ([{:type :file :val f} context]  {:literal (stencil/render-file f context)})
  ([{:type :literal :val s} _] {:literal s})
  ([{:type :s3 :val m} context] {:s3 m}))

(defn to-byte-array
  "Converts input into a byte-array, accepts same input as io/copy."
  [input]
  (let [o (ByteArrayOutputStream.)]
    (io/copy input o)
    (.toByteArray o)))

(defn s3-file
  "Gets an input stream for an s3 files, takes keys :bucket :key
   and optionally :access-key and :secret-key. If no credentials
   are provided it will attempt to get them from the machine."
  [o]
  (-> (or (-> (select-keys o [:access-key :secret-key])
              (#(when-not (empty? %) %)))
          (aws/get-credentials))
      (s3/get-object (:bucket o) (:key o))
      (:content)))

(def content-fns {:literal identity
                  :s3 s3-file})

(defn contents-from-wire [c]
  (->> ((-> c keys first content-fns)
        (-> c vals first))
       to-byte-array))

(defmethod g/to-wire "file" [node]
  (-> node
      (update-in [:attrs] assoc :contents
                 (render-template (:template (:attrs node)) ((get (:attrs node) :context-fn (fn [])))))
      (update-in [:attrs] dissoc :template :context-fn)))

(defmethod g/from-wire "file" [node]
  (-> (assoc-in node [:attrs :stat] (stat (-> node :attrs :pathname)))
      (update-in [:attrs :contents] contents-from-wire)))

(defmethod g/from-wire "directory" [node]
  (assoc-in node [:attrs :stat] (stat (-> node :attrs :pathname))))

(defn fs-node-checker [node my-type]
  (let [pathname (-> node :attrs :pathname)
        ftype (file-type pathname)
        exists? (fs/exists? pathname)]
    ((m/match [exists? my-type ftype]
              [true _ "unknown"]             (constantly {:status :error :reason
                                                          (str "Don't know what to do with path " pathname)})
              [true "file" "file"]           (node-path chown? chmod? write?)
              [true "file" _]                (node-path delete create chown chmod write)
              [true "directory" "directory"] (node-path chown? chmod?)
              [true "directory" _]           (node-path delete create-dir chown chmod)
              [false "file" _]               (node-path create chown chmod write)
              [_ _ _]                        (node-path create-dir chown chmod))
     node)))

(defmethod g/check-node "file" [node _ s]
  (fs-node-checker node "file"))

(defmethod g/check-node "directory" [node _ _]
  (fs-node-checker node "directory"))

(defmethod g/from-wire "file-permissions" [node]
  (assoc-in node [:attrs :stat] (stat (-> node :attrs :pathname))))

(defop ensure-exists [pathname]
  (if (fs/exists? pathname)
    (u/ok)
    (u/error (str pathname " does not exist."))))

(defmethod g/check-node "file-permissions" [node _ _]
  ((node-path ensure-exists chown? chmod?) node))
