;; owner: marshall@readyforzero.com
;;
;; Partition a drive into various sizes and partition types.
;;
;; A drive spec is a map with keys
;; :drive - The drive identifier, such as \"sda\"
;; :partitions - A list of maps that define a partition.
;;
;; A parition map contains the keys
;; :unit - The unit that the partition size is defined in.
;;         Can be :kb :mb or :gb, defaults to :mb.
;; :size - The size of the partition, either a number or :fill.
;;         If :fill the partition will fill all free space after
;;         all other partitions have been added. There can only be
;;         one partition with size set to :fill.
;; :type - The partition type, can be :normal :swap :extended,
;;         defaults to :normal.
;;
;; Ex:
;; (require '[borg.state.types.provided :as p])
;; (def dd (p/drive "sdb"
;;            {:partitions [{:size 10}
;;                          {:size 10 :unit :gb}
;;                          {:size 20 :type :swap}
;;                          {:size 15}
;;                          {:size 10}
;;                          {:size 13}
;;                          {:size :fill}]
;;             :provides :drive-sdb}))

(ns borg.state.types.drive
  (:require [borg.internal.lang :refer [-!>>]]
            [borg.state.graph :as g]
            [borg.state.types.core :refer [defop sh-result node-path try-to]]
            [borg.state.types.internal.drive :as i]
            [clojure.java.shell :as sh]
            [clojure.string :as string]))

;; Converts all partition sizes to use :mb as their unit
;; size, and removes the :unit key. If :size is :fill gets set to nil.
;; Also replaces the value of :type with a vector with the type
;; Char code and the type id.
(defmethod g/to-wire "drive"
  [node]
  (let [drive-specs (:attrs node)]
    (assoc node :attrs
      (->> (:partitions drive-specs)
           (map i/update-size)
           (map i/update-type)
           (assoc drive-specs :partitions)))))

(defmethod g/from-wire "drive"
  [node]
  (let [drive-specs (i/with-sectors (:attrs node))]
    (assoc node :attrs
      (->> (i/prepare-specs (->> (:partitions drive-specs)
                               (sort-by :size)
                               (reverse))
                          (:sector-size drive-specs)
                          (:sectors drive-specs))
           (i/with-extended-partition (:sectors drive-specs))
           (assoc drive-specs :partitions)))))

(defop unmount [drive-id]
  (try-to (str "unmount drive " drive-id)
          ;;TODO
          true))

(defop partition-drive [spec]
  (sh-result
   (->> (:partitions spec)
        (map #(update-in % [:type] first))
        (i/partitions->sfdisk)
        (#(str "echo -e \""
               % "\" | sfdisk --unit S --force "
               (i/drive-id (:drive spec))))
        (sh/sh "bash" "-c"))))

(defn check-partitions [node]
  (let [spec (:attrs node)]
    (if-not (->> (:partitions spec)
                 (map #(update-in % [:type] last))
                 (= (i/get-drive-info (:drive spec))))
      [(unmount (:drive spec))
        (partition-drive spec)]
      [])))

(defmethod g/check-node "drive"
  [node _ _]
  (check-partitions node))
