;; Owner: wolfson@readyforzero.com
;; Utilities for creating input for dot.
(ns borg.state.internal.viz
  (:require [clojure.string :as str]))

(defn fix-name [n]
  (str/replace (name n) #"\"|-|\{|\}" "_"))

(defn flatten-edges [edgepairs]
  (mapcat (fn [[source destinations]]
            (if (empty? destinations)
              [[source nil]]
              (map (fn [d] [source d]) destinations)))
          edgepairs))

(defn fmt-dot [edges graph-name]
  (str "digraph " graph-name " {\n"
       (str/join "\n" (map (fn [[s d]]
                             (let [s (fix-name (name s))
                                   d (when d (fix-name (name d)))]
                               (if d
                                 (str s " -> " d)
                                 s))) edges))
       "\n}\n"))



(defn mk-op-record-contents [op]
  (str/join "\\n" (cons (str "OP:" (fix-name (:op op)))
                      (map (fn [[k v]] (str/replace (str (name k) "=" v)
                                                   #"\""
                                                   "\\\\\"")) (:args op)))))

(defn mk-op-records [ops]
  (str/join "|" (map mk-op-record-contents ops)))

(defn make-cluster-node [[node-name node-record-contents]]
  (str node-name "[label=\"{" node-name "|{"
       (str/replace node-record-contents #"\{|\}" "_") "}}\"];"))

(defn make-clusters [nodemaps & [prefix color]]
  (letfn [(cname [counter] (str "cluster_" prefix counter))]
    (first (reduce (fn [[clusters counter] nodemap]
                     (let [text (str/join "\n"
                                          [(:cluster-text clusters)
                                           (str "subgraph " (cname counter)
                                                " {\n"
                                                (when color (str "color = " color ";\n"))
                                                (when (> (count nodemap) 1) "label = \"run in parallel\";\n")
                                                "node[shape=record];\n"
                                                "rankdir = TB;\n"
                                                (str/join "\n" (map make-cluster-node nodemap))
                                                "\n}\n")])
                           node-name (first (keys nodemap))]
                       [{:cluster-text text
                         :names (conj (:names clusters) node-name)
                         :cnames (conj (:cnames clusters) (cname counter))} (inc counter)]))
                   [{:cluster-text ""
                     :names []
                     :cnames []} 1]
                   nodemaps))))

(defn join-cluster-names [names cnames]
  (map (fn [start end c-start c-end]
         (str start " -> " end " [ltail=" c-start " lhead=" c-end "];"))
       names
       (rest names)
       cnames
       (rest cnames)))

(defn make-error-record [{:keys [planned executed error]}]
  (let [executed-record-contents (map mk-op-record-contents executed)
        failed-record-contents (-> (drop (count executed) planned)
                                   first
                                   mk-op-record-contents
                                   (str "\\nFAILED: " error))
        not-reached-record-contents (map #(str (mk-op-record-contents %) "\\nNOT EXECUTED")
                                         (drop (inc (count executed)) planned))]
    (str/join "|" (concat executed-record-contents [failed-record-contents] not-reached-record-contents))))

(defn make-error-maps [errors]
  (->> errors
       (map (fn [[k errspec]]
              [(fix-name k) (make-error-record errspec)]))
       (into {})))

(defn make-action-node-map [actions]
  (map (fn [ops] (->> ops (map (fn [[k v]] [(fix-name k) (mk-op-records v)])) (into {}))) actions))

(defn make-graph [name clusters]
  (str "digraph " name " {\ngraph[compound=true];\n"
       "rankdir = LR;\n"
       (str/join "\n" (map :cluster-text clusters))
       "\n"
       (str/join "\n" (join-cluster-names (mapcat :names clusters)
                                          (mapcat :cnames clusters)))
       "\n}\n"))
