;; Owner: wolfson@readyforzero.com
;; Functions to run actions (as returned from from
;; borg.state.graph/check-nodes).
;;
;; An "action" here is a map that has a :fn key whose value is a
;; nullary function that does something interesting and returns a map
;; with a :status key whose value is either :ok or :error.
;;
;; An "action list" is a list of actions.
(ns borg.state.internal.core
  (:require [borg.state.util :as u]
            [clojure.algo.generic.functor :as f]
            [clojure.core.match :as m]
            [clojure.tools.logging :as lg]))

(def remove-fn #(dissoc % :fn))

(defn run-actions
  "Run the actions in an action-list sequentially.

   If any action fails, we return an error structure indicating what
   steps were successfully executed, what step the borglet stopped on,
   what steps were planned, and an error message. Otherwise, we return
   a success structure listing all the steps that were executed."

  [actions dry-run?]
  (let [action-names (apply str (interpose " " (map :op actions)))]
    (lg/info "Running actions: " action-names))
  (loop [remaining actions log []]
    (if-let [next (first remaining)]
      (do
        (lg/info "Running action: " (:op next))
        (if dry-run?
          (recur (rest remaining) (conj log (remove-fn next)))
          (m/match [((:fn next))]
                   [{:status :ok} :as res] (recur (rest remaining)
                                                  (conj log (assoc (remove-fn next) :log (:log res))))
                   [{:status :error :reason reason}] {:status :error
                                                      :error reason
                                                      :planned (map remove-fn actions)
                                                      :executed log
                                                      :stopped-on (remove-fn next)})))
      {:status :ok :actions log})))

(defn run-shutdowns [shutdowns dry-run?]
  (lg/info "running shutdowns for " (apply str (interpose " " (map first shutdowns))))
  (loop [remaining shutdowns log []]
    (if-let [next (first remaining)]
      (let [[node-name op] next]
        (if dry-run?
          (recur (rest remaining) (conj log {node-name (remove-fn op)}))
          (m/match [((:fn op))]
                   [{:status :ok} :as res] (recur (rest remaining)
                                                  (conj log {node-name (assoc (remove-fn op) :log (:log res))}))
                   [{:status :error :reason reason}]
                   (do (lg/info "FAILED:" reason)
                     (let [removed (->> shutdowns
                                        (map (fn [[name op]]
                                               {name (remove-fn op)})))]
                       (lg/info "REMOVED:" removed)
                       {:status :error
                        :error reason
                        :planned removed
                        :executed log
                        :stopped-on {node-name (remove-fn op)}})))))
      {:status :ok :shutdowns log})))

(defn run-action-lists
  "Warning: here be terminology.

   Run a list of action maps, stopping at the first error.

   An action map is a map from node name to action specs for the node.

   An action spec is a map with an :actions key whose value is a list
   of actions, and optionally a :shutdown-first key whose value is a
   single operation to be run *before* traversing the graph and
   performing the actions in each node's :action list.

   Each action list in the vals of an action map can be run in
   parallel with the others.

   Each action map in a list of action maps must be run sequentially,
   one after the other.

   We run the shutdown-first actions, if any, from the top down, then
   run the actions for each node, in parallel where possible, from the
   bottom up.

   If any action fails, we stop.

   If we succeed in executing everything, a success response is
   returned detailing what happened. This response should be
   identical, except for log messages, to a dry-run response. If an
   error is encountered, we return an error response.

   The structure of each type of response is described by example in
   borg.state.test-core."
  
  [action-maps dry-run?]
  (let [shutdowns (remove (comp nil? second) (reverse (mapcat #(f/fmap :shutdown-first %) action-maps)))
        shutdown-result (if-not (empty? shutdowns)
                          (run-shutdowns shutdowns dry-run?)
                          (u/ok))
        planned (->> action-maps
                     (remove empty?)
                     (map #(f/fmap :actions %))
                     (mapv #(f/fmap (partial map remove-fn) %)))]
    (if-not (u/ok? shutdown-result)
      {:status :error
       :shutdowns shutdown-result
       :error-location :shutdowns
       :oks nil
       :executed nil
       :errors nil
       :planned planned
       :dry-run? dry-run?}
      (assoc 
          (loop [remaining-maps action-maps log []]
            (if-let [next-map (first remaining-maps)]
              (let [results (reduce merge {} (pmap (fn [[node-name action-spec]]
                                                     (let [res (run-actions (:actions action-spec) dry-run?)]
                                                       {node-name res}))
                                                   next-map))]
                (cond
                 (empty? results) (recur (rest remaining-maps) log)
                 (every? u/ok? (vals results)) (recur (rest remaining-maps) (conj log (f/fmap :actions results)))
                 :else (let [{oks true errors false} (group-by (comp u/ok? second) results)
                             oks (into {} oks)
                             errors (into {} errors)]
                         {:status :error
                          :error-location :actions
                          :errors  errors
                          :oks oks
                          :executed log
                          :planned planned
                          :dry-run? dry-run?})))
              {:status :ok :actions log :dry-run? dry-run?}))
        :shutdowns (:shutdowns shutdown-result)))))
