;; owner: marshall@readyforzero.com
;; Installs packages that the state lib depends on, either using the
;; native package managers or from a tbz2 file in resources.

(ns borg.state.init
  (:require [borg.state.core :as state]
            [borg.state.graph :as g]
            [borg.state.internal.core :as i]
            [borg.state.types.provided :as p]
            [clojure.java.io :as io]
            [clojure.java.shell :as sh]
            [clojure.tools.logging :as lg]))

(defn lg-sh [& cmd]
  (lg/info "Running CMD:" cmd)
  (doseq [l (apply sh/sh cmd)]
    (lg/info l)))

(defn apply-nodes
  "Takes a variable number of state nodes, such as (p/file ...),
   and applies them to the local machine, and returns the :status key
   from the results."
  [& nodes]
  (-> (apply state/merge-graphs nodes)
      (g/simulate-send)
      (g/check-nodes)
      (i/run-action-lists false)
      :status))

(defn install-from-file
  "Installs runit from runit-rpm.tbz2 in resources."
  []
  (lg/info "runit failed to install via package manager, installing via tar")
  (when (-> (apply-nodes
             (p/package "rpm-build" "4.8.*" {:provides :rpm-build})
             (p/package "rpmdevtools" "7.*" {:provides :rpmdevtools}))
            (= :ok))
    (-> (io/resource "runit-rpm.tbz2")
        (io/file)
        (io/copy (io/file "/tmp/runit-rpm.tbz2")))
    (lg-sh "tar" "-C" "/tmp" "-xjf" "/tmp/runit-rpm.tbz2")
    (lg-sh "/tmp/runit-rpm/build.sh")
    (lg-sh "rpm" "-i" "~/rpmbuild/RPMS/*/*.rpm")))

(defn install-from-package []
  (-> (p/package "runit" "2.1.1" {:provides :runit})
      (apply-nodes)))

(defn install-runit
  "Installs runit if not already installed."
  []
  (lg/info "Install runit if not already installed")
  (when (= :error (install-from-package)) 
    (install-from-file)))