;; Owner: wolfson@readyforzero.com
;; Utility functions and multimethods related to state graphs and
;; state nodes.
(ns borg.state.graph
  (:require [babbage.util :as graph-util]
            [borg.state.util :as u]
            [clojure.algo.generic.functor :as functor]
            [clojure.set :as set]))

(defn node-attr [n attr]
  (-> n :attrs attr))

(defn node-type [node]
  (node-attr node :type))

(defmulti to-wire
  "This function will be called on nodes of a particular type prior to
   their being serialized into JSON for transmission to the borglet."
  node-type)

(defmethod to-wire :default [node] node)

(defmulti from-wire
  "This function will be called on nodes of a particular type after
   their being deserialized from JSON when received by a borglet."
  node-type)

(defmethod from-wire :default [node] node)

(defmulti check-node
  "Returns the actions, if any, necessary for this node.

   Additional arguments are a map from node names to nodes, and list
   of maps of node names to actions previously computed to be required
   for those nodes."
  (fn [node nodemap earlier-actions] (node-type node)))

(defn check-node-ops [node nodemap earlier-actions]
  (let [result-map (check-node node nodemap earlier-actions)
        result-map (if (map? result-map) result-map {:actions result-map})
        allowed-op-types (set (map keyword (:produces-ops node)))]
    (when-let [bad-op (some #(when-not (allowed-op-types %) %) (map :op (:actions result-map)))]
      (throw (Exception. (format "Node %s, type %s, produced unrecognized op %s"
                                 node (node-type node) bad-op))))
    result-map))

(defn check-nodes
  "From a deserialized representation of the graph, return a
   representation of actions. The deserialized input is a map with a
   nodemap key and a layers key: the nodemap maps node names to nodes;
   \"layers\" is a list of lists of nodes s.t. each list contains
   nodes that can be processed independently of each other. To each
   node there might correspond multiple actions, so the return value
   of this function is a list of maps of node names to maps with the
   form {:actions list-of-actions :pre-shutdown shutdown-fn-or-nil}."
  [{:keys [layers nodemap]}]
  (reduce (fn [layer-actions layer]
            (conj layer-actions
                  (reduce (fn [acc node]
                            (let [action-spec (check-node-ops node nodemap layer-actions)]
                              (if (or (seq (:actions action-spec)) (:shutdown action-spec))
                                (merge acc {(keyword (:provides node)) action-spec})
                                acc)))
                          {}
                          layer)))
          []
          layers))

(defn requires? [node1 node2]
  (some #(= (:provides (:node node2)) %) (:requires (:node node1))))

(defn graph-to-wire [state]
  ;;; traverse the graph once on the client side: spares each machine
  ;;; having to do so and catches cycles etc. before sending the graph
  ;;; out.
  (let [nodes (map to-wire (vals (:nodemap state)))]
    (graph-util/layers nodes)))

(defn graph-from-wire [layers]
  (let [layers (map (partial map from-wire) layers)
        nodemap (->> (apply concat layers)
                     (map (juxt :provides identity))
                     (into {}))]
    {:nodemap nodemap :layers layers}))

(defn simulate-send
  "Take a graph generated by the borglet itself and convert it to a
   form that simulates transmission from a client to the borglet. This
   calls to-wire and from-wire on the nodes, but does *not* send the
   graph through JSON."
  [graph]
  (graph-from-wire (graph-to-wire graph)))

(defn merge-graphs
  ([graph1] graph1)
  ([graph1 graph2]
     (if (requires? graph1 graph2)
       (if (requires? graph2 graph1)
         (throw (Exception. (str "Circular dependencies: " graph1 graph2)))
         {:node (:node graph1)
          :nodemap (merge (:nodemap graph1) (:nodemap graph2))})
       {:node (:node graph2)
        :nodemap (merge (:nodemap graph2) (:nodemap graph1))}))
  ([graph1 graph2 & graphs] (apply merge-graphs (merge-graphs graph1 graph2) graphs)))
