;; owner: marshall@readyforzero.com
;; In memory registry for testing.

(ns borg.registry.virtual
  (:require [borg.registry.interface :refer :all]
            [clojure.string :as string])
  (:refer-clojure :exclude [remove])
  (:import java.util.Date))

(defn ip-port [ip port]
  (str ip ":" port))

(defn insert [m classifiers ip-port]
  (println "inserting")
  (if classifiers
    (update-in m classifiers assoc ip-port (Date.))
    (assoc m ip-port (Date.))))

(defn remove [m classifiers ip-port]
  (if classifiers
    (update-in m classifiers #(dissoc % ip-port))
    (dissoc m ip-port)))

(defn safe-join [char a b]
  (if a
    (string/join char [a b])
    b))

(defn names [m query fresh-fn classifier]
  (let [m2 (if (seq query)
             (select-keys m (take 1 query))
             m)
        ips (->> (keys m)
                 (filter string?))
        ip-objects (if (not (seq query))
                     (->> (select-keys m ips)
                          (filter fresh-fn)
                          (map (fn [[k t]] (safe-join ":" classifier k))))
                     [])
        query (rest query)]
    (flatten [ip-objects
              (map (fn [[k v]]
                     (->> (name k)
                          (safe-join ":" classifier)
                          (names v query fresh-fn)))
                   (apply dissoc m2 ips))])))

(defn fresh? [now cutoff [k v]]
  (->> (.getTime v)
       (- now)
       (> cutoff)))

(defrecord Virtual [db]
  IRegistry
  (register [_ ip port classifiers]
    (swap! (:db _) insert classifiers (ip-port ip port)))

  (unregister [_ ip port classifiers]
    (swap! (:db _) remove classifiers (ip-port ip port)))

  (get-names [_ cutoff query excluded?]
    (names @(:db _)
           query
           (let [now (.getTime (Date.))]
             (if excluded?
               #(not (fresh? now cutoff %))
               #(fresh? now cutoff %)))
           nil))
  
  (purge-names [_ cutoff]
    (let [names (get-names _ cutoff nil true)
          parts (map #(string/split % #"-") names)
          classifiers (map #(when (< 1 (count %))
                              (-> (first %)
                                  (string/split #":")
                                  (->> (map keyword))))
                           parts)]
      (map #(apply swap! (:db _) remove %) (zipmap classifiers (map last parts))))))

(defn init [options]
  (Virtual. (atom {})))