;; owner: marshall@readyforzero.com
;; Defines a registry that uses s3 as the data store.

(ns borg.registry.s3
  (:require [borg.aws.core :as aws]
            [borg.registry.interface :refer :all]
            [clojure.string :as string]
            [aws.sdk.s3 :as s3])
  (:import java.util.Date))

(defn make-key [ip port classifiers]
  (->> (concat classifiers [ip (str port)])
       (map name)
       (string/join ":")))

(defn fresh? [now cutoff obj]
  (->> (obj :metadata)
       :last-modified
       .getTime
       (- now)
       (> cutoff)))

(defrecord S3 [bucket credentials]
  IRegistry
  (register [_ ip port classifiers]
    (s3/put-object (:credentials _) (:bucket _) (make-key ip port classifiers) ""))

  (unregister [_ ip port classifiers]
    (s3/delete-object (:credentials _) (:bucket _) (make-key ip port classifiers)))

  (get-names [_ cutoff query excluded?]
    (let [now (-> (Date.) .getTime)
          filter-fn (if excluded? #(not (fresh? now cutoff %))  #(fresh? now cutoff %))
          query (->> (map name query)
                     (string/join ":"))]
      (->> (s3/list-objects (:credentials _) (:bucket _) {:prefix query})
           :objects
           (filter filter-fn)
           (map :key))))

  (purge-names [_ cutoff]
    (->> (get-names _ cutoff nil true)
         (map #(s3/delete-object (:credentials _) (:bucket _) %)))))

(defn init
  "Config must have the following keys
   :bucket"
  [{:keys [bucket]}]
  (let [creds (aws/get-credentials)]
    (if (and bucket creds)
      (S3. bucket creds)
      (throw (Exception. "bucket and credentials can not be nil")))))