;; owner: marshall@readyforzero.com
;; Functions for common patterns for borg modules.
;; Each module has two vars, one is a map of instance types,
;; the second contains an instantiated instance.
;;
;; These functions provide a common way of initializing those
;; variables and modifying them.

(ns borg.internal.module
  (:require [borg.config :as config]
            [clojure.tools.logging :as lg]))

(def mod-types (symbol "module-types"))
(def instance (symbol "module-instance"))

(defmacro create-vars
  "Creates the types var with the 'types' map,
   and an the instance var."
  [types]
  `(do (def ~mod-types (atom ~types))
       (defonce ~instance (atom nil))))

(defmacro add-type
  "Adds a new instance type init function for 'key'."
  [key fn]
  `(swap! ~mod-types assoc ~key ~fn))

(defmacro get-instance
  "Returns the current type instance, or if nil creates the a new instance
   using the type and options specified in borg.config under 'config-key'.
   If :type or :options under 'config-key' are nil their default value
   will be used."
  [config-key default & [opt-default]]
  `(if-let [i# (deref ~instance)]
     i#
     (reset! ~instance (((config/get [~config-key :type] ~default keyword)
                         (deref ~mod-types))
                        (config/get [~config-key :options] ~opt-default)))))

(defmacro set-instance
  "Takes a keyword or a function and a map of options that will be passed
   to the instance init function. If the type is a keyword, retrieves
   the init function from the types var."
  [type options]
  `(reset! ~instance ((if (keyword? ~type)
                        (~type (deref ~mod-types))
                        ~type)
                      ~options)))
