;; owner: marshall@readyforzero.com
;;
;; Responsible for the list of borglet users and what level they have
;; permission to run.
;; The different levels corraspond to the different type of handlers
;; (defhandler => :none, defauthedhandler => :auth,
;; defsuperhandler => :super).
;; Higher levels can run all levels lower than them (:super > :none).
;;
;; In addition to maintaing the user to permission list in memory, it
;; also creates and removes unix users for each borglet user when they
;; have been added or deleted.
;;
;; -- Config --
;; Key :user-store
;; Sub-keys
;; :type - (string) user store type (ex: s3, virtual), default virtual.
;; :options - (map) options that will be passed to the init fn.

(ns borg.borglet.users.core
  (:refer-clojure :exclude [remove sync])
  (:require (borg.borglet.users
              [interface :as in]
              [virtual :as virtual]
              [s3 :as s3])
            [borg.internal.module :as m]
            [clojure.java.io :as io]
            [clojure.java.shell :as sh]
            [clojure.set :as set]
            [clojure.string :as string]))

(def levels [:auth :super])

;; see borg.internal.module for behavior
(m/create-vars {:virtual virtual/init
                :s3 s3/init})

(defn add-store-type! [key fn]
  (m/add-type key fn))

(defn set-permission-store! [key & [options]]
  (m/set-instance key options))

(defn get-store []
  (m/get-instance :user-store :virtual))
;; end module section

(defn create-unix
  "Create a unix user with username and home dir borg-username."
  [username]
  (sh/sh "useradd" "--create-home"
         "--home-dir" (str "/home/borg-" username)
         username))

(defn delete-unix
  "Delete a unix user"
  [username]
  (sh/sh "userdel" "--remove" username))

(defn resolve-users
  "Adds any users that do not yet have unix accounts
   and removes old unix accounts that are no longer in
   the internal user list. "
  [users]
  (let [users (set (keys users))
        existing (->> (io/file "/home")
                      (.listFiles)
                      (map #(.getName %))
                      (filter #(.regionMatches % 0 "borg-" 0 5))
                      (map #(.substring % 5))
                      (set))
        add (set/difference users existing)
        remove (set/difference existing users)]
    (doseq [u add]
      (create-unix u))
    (doseq [u remove]
      (delete-unix u))))

(defn sync-users
  "Refreshes the user list from s3 and creates and deletes
   unix accounts so that the machine is in sync with s3."
  []
  (->> (get-store)
       (in/sync)
       (in/get-all)
       (resolve-users)))

(defn can-run?
  "Returns true if that username has permission
   to run that level, false otherwise."
  [username level]
  (in/can-run? (get-store) levels username level))
