;; owner: marshall@readyforzero.com
;; Internal functions for running a borglet:
;; - Starting the borglet transport
;; - Activating the registry
;; - Fns/handlers for updating a borglet
(ns borg.borglet.internal.core
  (:require [borg.auth.lockout :as lockout]
            [borg.aws.ec2 :as ec2]
            [borg.borglet.config :as b-config]
            [borg.borglet.handler.core :as handler]
            [borg.borglet.handler.handlers]
            [borg.borglet.util.io :as util-io]
            [borg.config :as config]
            [borg.internal.lang :refer [-?>]]
            [borg.internal.util :as util]
            [borg.registry.core :as reg]
            [borg.state.core :as state]
            [borg.state.graph :as graph]
            [borg.state.internal.core :as state-i]
            [borg.state.types.provided :as p]
            [borg.transport.core :as trans]
            [clojure.set :as set]
            [clojure.java.io :as io]
            [clojure.tools.logging :as lg])
  (:import [java.util.concurrent ScheduledThreadPoolExecutor TimeUnit]))

(defonce scheduler (atom nil))
(defonce server (atom nil))
(defonce unreg-fn (atom nil))

(def default-options
  {:port 2323
   :registry-interval 30})

(defn init-etc
  "Creates the directories /etc/borg and /etc/borg/service."
  []
  (-> (state/merge-graphs
       (p/directory "/etc/borg"
                    {:provides :etc-borg
                     :user "root"
                     :group "root"
                     :permissions "0755"})

       (p/directory "/etc/borg/service"
                    {:provides :etc-borg-service
                     :requires [:etc-borg]
                     :user "root"
                     :group "root"
                     :permissions "0755"}))
      (graph/simulate-send)
      (graph/check-nodes)
      (state-i/run-action-lists false)))

(defn register
  "Register the borglet with the current registry using the
   ip, port, and classifiers from config."
  []
  (let [classifiers (config/get [:registry :classifiers])
        ip (config/get [:registry :ip] (reg/ip) nil true)
        port (config/get :port)]
    (try
      (reg/register ip port classifiers)
      (lg/info "Checked in to registry with " ip port classifiers)
      (catch Exception e
        (lg/info "Failed registration with error" (.getMessage e))))))

(defn unregister
  "Unregister the borglet with the current registry using the ip,
   port and classifiers from config. Optionally override the classifiers."
  [& [my-classifiers]]
  (let [{classifiers :classifiers
         ip :ip} (config/get :registry)
         port (config/get :port)]
    (reg/unregister ip port (or my-classifiers classifiers))))

(defn start-registry []
  (when-let [type (config/get [:registry :type])]
    (when-let [tags (config/get [:aws :lookup-classifiers])]
      (config/set :classifiers (ec2/get-tags tags)))
      (lg/info "Set classifiers from aws")
    (let [interval (config/get [:registry :interval]
                               (:registry-interval default-options)
                               #(util/unit->ms % :minute)
                               true)]
      (->> (doto (ScheduledThreadPoolExecutor. 1)
             (.scheduleAtFixedRate register 0 interval TimeUnit/MILLISECONDS))
           (reset! scheduler))
      (b-config/add-listener
       (fn [old new]
         (when (not= (:classifiers old) (:classifiers new))
           (register)
           (unregister (:classifiers old))))
       :registry))))

(defn startup
  "See borg.borglet.core for option docs."
  []
  (lg/info "starting with options" @config/state)
  (->> (config/get :port (:port default-options) #(Integer. %) true)
       (trans/borglet-start handler/call-handler)
       (reset! server))
  (lg/info "Borglet online")
  (start-registry))

(defn remove-update-files []
  (lg/info "removing files")
  (for [fname ["updating"
               "old.rev"
               "new.rev"]
        :let [file (io/as-file (str "../../" fname))]]
    (do
      (lg/info file)
      (when (.exists file)
        (io/delete-file file)))))

(defn shutdown
  "If using a registry stops the scheduled checkins and unregisters
   its self from the registry, then shuts down the transport."
  []
  (when (config/get [:registry :type])
    (.shutdown @scheduler)
    (unregister))
  (trans/borglet-stop @server))

(defn stop
  "Helper for use by the handlers to shutdown."
  []
  (future (shutdown))
  (shutdown-agents)
  true)

;;;;;;;;;;;;;; Borglet Specific Handlers ;;;;;;;;;;;;;;;
(handler/defauthedhandler shutdown
  "Stops a borglet."
  [options]
  (stop))

(handler/defhandler current-revision
  "Returns the current commit that borglet is on"
  [options]
  (util-io/git-revision "root"))

(handler/defauthedhandler update-borglet
  "Updates a borglet to the supplied revision using the specified repo.
  Ex: (run :update-borglet {:repo-url \"git@github.com:..\"
                            :commit \"shduf26d...\"})"
  [{:keys [repo-url commit]}]
  (let [cur-commit (util-io/git-revision "root")]
    (util-io/git-deploy-revision {:repo repo-url
                                  :commit commit
                                  :user (config/get :user "root")
                                  :ssh-key (config/get :ssh-key)
                                  :revisions-dir "../"})
    (spit "../../old.rev" cur-commit)
    (spit "../../new.rev" commit))
  (stop))
