;; Must be run as a user who has su access.
(ns borg.borglet.core
  (:require [borg.borglet.internal.core :as i]
            [borg.borglet.config :as b-conf]
            [borg.config :as config]
            [borg.state.init :as state]))

(defn start
  "Starts a borglet with the specified options.
   Takes either a path to the json encoded options file,
   or a map of the options directly. If no arguments are
   supplied it will try and load the options from the
   default location.

   -- Optional --
   :port - The port to run on, defaults to 2323.
   :user - The name of the user the borglet will act as.

   See each borg module for options documentation:
   borg.auth, borg.transport, borg.registry, borg.borglet.users"
  [& [options]]
  (i/init-etc)
  (state/install-runit)
  (config/init options)
  (i/startup)
  (.addShutdownHook (Runtime/getRuntime) (Thread. i/shutdown))
  (doall (i/remove-update-files)))

(defn stop []
  (i/shutdown))

(defn add-listener
  "Adds a function that gets called when the borglet config changes.
   Takes an optional key to listen for changes on. If the key is supplied
   only the value of the key will be sent to the function, otherwise the entire
   config map will be sent.

   Returns a key that you can use to remove the listener."
  [func & [key]]
  (b-conf/add-listener func key))

(defn remove-listener
  "Remove a listener by the key that was returned when adding it."
  [key]
  (b-conf/remove-listener key))

(defn -main [& [port user]]
  (let [opt (or (System/getenv "BORG_JSON_CONFIG")
                (when (and port user)
                  {:port port
                   :user user}))]
    (start opt)))
