;; owner: marshall@readyforzero.com
;; AWS specific functions
;;
;; -- Config --
;; Key :aws-credentials
;; Sub-keys
;; :access-key - (optional) aws access key
;; :secret-key - (optional) aws secret key

(ns borg.aws.core
  (:require [borg.config :as config])
  (:import [com.amazonaws.auth
            AWSCredentials
            AWSSessionCredentials
            InstanceProfileCredentialsProvider]))

(defrecord BorgAWSCredentials [access-key secret-key]
  AWSCredentials
  (getAWSAccessKeyId [_] (:access-key _))
  (getAWSSecretKey [_] (:secret-key _)))

(defrecord BorgSessionCredentials [access-key secret-key session-token]
  AWSCredentials
  AWSSessionCredentials
  (getAWSAccessKeyId [_] (:access-key _))
  (getAWSSecretKey [_] (:secret-key _))
  (getSessionToken [_] (:session-token _)))

(defn init-creds
  "Creates an instance of BorgAWSCredentials if given an instance
   of AWSCredentials or a map with keys :access-key :secret-key.
   Creates an instance of BorgSessionCredentials if given AWSSessionCredentials.
   Doing it this way allows us to interchangeably use the credentials
   object with the aws java sdk and our own code that expects a map
   with the same keys as above, including :session-token if using
   BorgSessionCredentials."
  [arg]
  (let [{ac :access-key sc :secret-key}
        (if (instance? AWSCredentials arg)
          {:access-key (.getAWSAccessKeyId arg)
           :secret-key (.getAWSSecretKey arg)}
          arg)]
    (if (instance? AWSSessionCredentials arg)
      (BorgSessionCredentials. ac sc (.getSessionToken arg))
      (BorgAWSCredentials. ac sc))))

(defn get-credentials
  "Returns a BorgAWSCredentials or BorgSessionCredentials instance,
   with keys :access-key, :secret-key, and if BorgSessionCredentials,
   :session-token.
   If no credentials were found in borg.config attempts to
   get temporary credentials from the machine."
  []
  (->  (if-let [creds (config/get [:aws :credentials])]
         creds
         (-> (InstanceProfileCredentialsProvider.)
             (.getCredentials)))
       (init-creds)))
