;; owner: marshall@readyforzero.com
;; Create, update, and delete cloudformation stacks using template files.

(ns borg.aws.cloudformation
  (:require [clojure.java.io :as io])
  (:import [com.amazonaws.services.cloudformation.model
            UpdateStackRequest
            CreateStackRequest
            DeleteStackRequest
            Parameter
            Capability]
           com.amazonaws.services.cloudformation.AmazonCloudFormationClient
           com.amazonaws.auth.BasicAWSCredentials))

(defn make-stack-obj
  "Creates a stack request (either create or update) with the template body,
   stack name, and an optional legacy chef-role."
  [create? template-body stack-name & [chef-role]]
  (-> (if create? (CreateStackRequest.) (UpdateStackRequest.))
      (.withParameters [(-> (Parameter.)
                           (.withParameterKey "ChefRole")
                           (.withParameterValue chef-role))])
      (.withTemplateBody template-body)
      (.withStackName stack-name)
      (.withCapabilities ["CAPABILITY_IAM"])))

(defn make-client
  "Creates cloudfront client to either create, update, or delete stacks."
  [credentials]
  (-> (BasicAWSCredentials. (:access-key credentials) (:secret-key credentials))
      (AmazonCloudFormationClient.)))

(defn create-or-update
  "Create or update a stack.
   type - can be :create or :update.
   credentials - should be a map with :access-key and :secret-key.
   stack-name - the name of the stack
   template-body - the contents of the template..
   chef-role - optionally include for legacy templates."
  [type credentials stack-name template-body & [chef-role]]
  (let [stack-obj (make-stack-obj (= :create type)
                                  template-body
                                  stack-name
                                  chef-role)
        client (make-client credentials)]
    (if (= :create type)
      (.createStack client stack-obj)
      (.updateStack client stack-obj))))

(defn delete-stack
  "Deletes a stack, takes the same credentials format as create-or-update."
  [credentials stack-name]
  (-> (make-client credentials)
      (.deleteStack (-> (DeleteStackRequest.)
                        (.withStackName stack-name)))))