;; owner: marshall@readyforzero.com
;; Used for authentication on both borglets and clients.
;; If you do not specify an authenticator with set-authenticator! the
;; None authenticator will be used, which requires no authentication.
;;
;; -- Config --
;; Key :auth
;; Sub-keys
;; :type - (string) authenticator type (ex: ssh, virtual), default none.
;; :options - (map) options that will be passed to the auth init fn.
;; :max-attempts - (optional) The maximum number of login attempts
;;                 before being locked out, defaults to 15.
;; :lockout-duration - (optional) Lockout duration in minutes, default 5.

(ns borg.auth.core
  (:require [borg.auth.interface :as in]
            [borg.auth.lockout :as lockout]
            [borg.auth.none :as none]
            [borg.auth.ssh :as ssh]
            [borg.internal.module :as m]
            [clojure.tools.logging :as lg]))

;; see borg.internal.module for behavior
(m/create-vars {:none none/init
                :ssh ssh/init})

(defn add-authenticator-type! [key fn]
  (m/add-type key fn))

(defn get-authenticator []
  (m/get-instance :auth :none))

(defn set-authenticator! [auth & [config]]
  (m/set-instance auth config))
;; end module section

(defn add-user
  "Adds a user to the borglet. See your authenticator
   documentation for the possible options."
  [user options]
  (in/add-user (get-authenticator) user options))

(defn add-users
  "Add multiple users at once, takes a list
   of maps with keys :user :options."
  [users]
  (in/add-users (get-authenticator) users))

(defn modify-user
  "Modify a user, such as changing a password or a public key.
   See your authenticator for the possible options."
  [user options]
  (in/modify-user (get-authenticator) user options))

(defn remove-user
  "Remove a user from a borglet."
  [user]
  (in/remove-user (get-authenticator) user))

(defn authenticate
  "Returns a map that will be sent to the borglet
   to authenticate using the options specified when
   creating the authenticator.
   The map contains the keys
   :identifier - a unique identifier for the user
   :data - a map of any additional paramters used by
           the authenticator."
  []
  (in/authenticate (get-authenticator)))

(defn validate-attempt
  "Checks the map returned from authenticate, returns
   an error message for a failure, and nil
   if successfully authenticated."
  [data]
  (lockout/auth-attempt
    data
    (in/validate-attempt (get-authenticator) data)))
