;; owner: marshall@readyforzero.com
;; Defines a registry that uses s3 as the data store.

(ns borg.registry.s3
  (:require [borg.registry.interface :refer :all]
            [clojure.string :as string]
            [aws.sdk.s3 :as s3])
  (:import java.util.Date))

(defn make-key [ip port classifiers]
  (-> (map name classifiers)
      (->> (string/join ":"))
      (str "-" ip ":" port)))

(defn fresh? [now cutoff obj]
  (->> (obj :metadata)
       :last-modified
       .getTime
       (- now)
       (> cutoff)))

(defrecord S3 [bucket access-key secret-key]
  IRegistry
  (register [_ ip port classifiers]
    (s3/put-object _ (:bucket _) (make-key ip port classifiers) ""))

  (unregister [_ ip port classifiers]
    (s3/delete-object _ (:bucket _) (make-key ip port classifiers)))

  (get-names [_ cutoff query excluded?]
    (let [now (-> (Date.) .getTime)
          filter-fn (if excluded? #(not (fresh? now cutoff %))  #(fresh? now cutoff %))]
      (->> (s3/list-objects _ (:bucket _) {:prefix query})
           :objects
           (filter filter-fn)
           (map :key))))

  (purge-names [_ cutoff]
    (->> (get-names _ cutoff "" true)
         (map #(s3/delete-object _ (:bucket _) %)))))

(defn init
  "Config must have the following keys
   :bucket :access-key :secret-key"
  [config]
  (S3. (:bucket config)
       (:access-key config)
       (:secret-key config)))