(ns borg.borglet.handler.core)

(def handlers (atom {}))
(def user (atom nil)) ;; a hack until an actual user if specified from the client
(def ^:dynamic *user* nil)

;;****************************
;; create an error or success response
;;****************************

(defrecord Response [status details error])

(defn make-response [success & [details error]]
  (->Response success details error))

(defn auth-required []
  (make-response :auth-required))

(defn auth-fail []
  (make-response :auth-failed))

(defn auth-success []
  (make-response :auth-success))

(defn make-error
  [msg & [details]]
  (make-response :fail details msg))

(defn make-success [details]
  (make-response :ok details))

(defn result->map [r]
  {:status (:status r)
   :details (:details r)
   :error (:error r)})

(defn ->result [obj]
  (-> (if (= Response (type obj))
        obj
        (make-success obj))
      (result->map)))

(defmacro make-handler [handle auth & args]
  (let [{all :a doc :d}
        (if (-> args first string?)
          {:d (first args) :a (rest args)}
          {:d nil :a args})
        params (first all)
        body (rest all)]
     `(swap! handlers assoc ~(str handle)
             {:fn (fn ~params ~@body)
              :doc ~doc
              :auth ~auth})))

;;*****************************
;; register and call handlers
;;*****************************

(defmacro defhandler
  "Creates a handler function.
   Params must be a vector of one arg, whose value will be a map."
  [handle & body]
  `(make-handler ~handle false ~@body))

(defmacro defauthedhandler
  "Same as defhandler except the client must be authenticated."
  [handle & body]
  `(make-handler ~handle true ~@body))

(defn call-handler
  "Ex: (call-handler {:handler \"shell\" :options {:command \"ls ~/\"})"
  [{:keys [handler options]} authed]
  (binding [*user* user]
  (-> (try
        (if-let [handler-map (get @handlers handler)]
          (if (or authed (not (:auth handler-map)))
            ((:fn handler-map) options)
            (auth-required))
          (make-error (str "There is no handler with the name " handler)))
        (catch Exception e (do (doall (map println (.getStackTrace e)))
                               (make-error (.getMessage e)))))
      (->result))))

(defhandler handlers
  "Returns a map of the handlers with their
   doc string and if they require authentication."
  [options]
  (let [h @handlers]
    (->> (vals h)
         (map (juxt :auth :doc))
         (zipmap (->> (keys h)
                      (map keyword))))))
