;; Must be run as a user who has su access.
(ns borg.borglet.core
  (:require [borg.borglet.config :as config]
            [borg.borglet.handler.core :as handler]
            [borg.borglet.handler.handlers]
            [borg.borglet.state :as state]
            [borg.borglet.util.io :as util-io]
            [borg.transport.core :as transport]
            [clojure.java.io :as io]
            [clojure.tools.logging :as lg]))

(def server (atom nil))

(defn remove-update-files []
  (lg/info "removing files")
  (for [fname ["updating"
               "old.rev"
               "new.rev"]
        :let [file (io/as-file (str "../../" fname))]]
    (do (println file)
    (when (.exists file)
      (io/delete-file file)))))

(defn start
  "Start a borglet listening on the specified, and who will perform
   actions as the specified user.
   Also accepts an optional function that will be called any time
   the borglet properties map changes. The function should accept 3 args,
   the ref to the actual map, the map before the change, and the map after
   the change."
  [port user & [state-change-handler]]
  (config/init user)
  (reset! handler/user user)
  (transport/set-transporter! :http)
  (reset! server (transport/borglet-start handler/call-handler (Integer. port)))
  (lg/info "Borglet online")
  (doall (remove-update-files)))

(defn stop []
  (future (transport/borglet-stop @server))
  (shutdown-agents)
  true)

(handler/defauthedhandler shutdown [options]
  (stop))

(handler/defhandler current-revision [options]
  (util-io/git-revision "root"))

(handler/defauthedhandler update-borglet [{:keys [repo-url commit]}]
  (let [cur-commit (util-io/git-revision "root")]
    (util-io/git-deploy-revision {:repo repo-url
                                  :commit commit
                                  :user (config/get :user)
                                  :ssh-key (config/get :ssh-key)
                                  :revisions-dir "../"})
    (spit "../../old.rev" cur-commit)
    (spit "../../new.rev" commit))
  (stop))

(handler/defhandler get-vars [options]
  @config/options)

(defn -main [& [port user]]
  (start port user))
