(ns reaction.healthcheck.component
  "Stateful components for running healthchecks in a component style system."
  (:require
    [clojure.spec.alpha :as s]
    [integrant.core :as ig]
    [reaction.healthcheck :as check]))

(defprotocol IHealthcheckRegistry
  (register [this k check-fn] "Registers a healthcheck.")
  (unregister [this k] "Unregisters a healthcheck."))

(defprotocol IHealthcheckRunner
  (run-checks! [this] "Runs checks in the registry.")
  (run-all! [this] "Runs checks in the registry and summarizes healthiness."))

(defrecord HealthcheckRunner [registry]
  IHealthcheckRegistry
  (register [this k check-fn]
    (swap! check/register (:registry this) k check-fn))
  (unregister [this k] (swap! check/unregister (:registry this) k))
  IHealthcheckRunner
  (run-checks! [this] (check/run-checks! @(:registry this)))
  (run-all! [this] (check/run-all! @(:registry this))))

(defn healthcheck-runner
  "Creates a healthcheck with an optional registry, which should be an atom."
  ([] (healthcheck-runner (atom {})))
  ([registry]
   {:pre [(instance? clojure.lang.Atom registry)]}
   (->HealthcheckRunner registry)))


;;
;; Integrant
;;

(s/def ::checks ::check/registry)
(s/def ::opts (s/keys :req-un [::check/checks]))

(defmethod ig/init-key :reaction.healthcheck/component
  [_ {checks :checks}]
  (if checks
    (healthcheck-runner (atom checks))
    (healthcheck-runner)))

(defmethod ig/halt-key! :reaction.healthcheck/component
  [_ _]
  nil)

(defmethod ig/pre-init-spec :reaction.healthcheck/component
  [_]
  ::opts)
