(ns reaction.healthcheck
  "Runs health checks for Clojure applications."
  (:require
    [clojure.spec.alpha :as s]
    [clojure.spec.gen.alpha :as gen]))

(s/def ::healthy? boolean?)
(s/def ::key keyword?)
(s/def ::description string?)
(s/def ::detail any?)
(s/def ::result (s/keys :req [::healthy?] :opt [::detail]))
(s/def ::check-fn (s/fspec :args (s/cat :nullary (s/cat)) :ret ::result))
(s/def ::check (s/keys :req [::description ::check-fn]))

(s/def ::result (s/keys :req [::healthy?]
                        :opt [::description ::detail ::check-fn]))

(s/def ::results (s/coll-of ::result :gen-max 8))

(s/def ::all-results (s/keys :req [::healthy? ::results]))

(s/def ::registry
  (s/with-gen (s/map-of ::key ::check)
    #(gen/bind (s/gen (s/map-of ::key ::check :gen-max 8))
       (fn [registry]
         (gen/return registry)))))

(defn register
  "Adds a check to the provided registry."
  [registry k check-fn]
  (conj registry {k check-fn}))

(s/fdef register
  :args (s/cat :registry ::registry :k ::key :check-fn ::check-fn)
  :ret  ::registry
  :fn   #(contains? (:ret %) (-> % :args :key)))

(defn unregister
  "Adds a check to the provided registry."
  [registry k]
  (dissoc registry k))

(s/fdef unregister
  :args (s/cat :registry ::registry :k ::key)
  :ret  ::registry)

(defn run-check!
  "Defensively runs a check."
  [curr k {:keys [::check-fn] :as v}]
  (if (not (fn? check-fn))
    (assoc curr k {::healthy? false
                   ::detail "Healthcheck check-fn is not a function."})
    (try
      ; check-fn returns a map. merge into check definition for result.
      (assoc curr k (into v (check-fn)))
      (catch Exception e
        (assoc curr k
          (into v {::healthy? false
                   ::detail (ex-info "Exception while running healthcheck." {} e)}))))))

(defn run-checks!
  "Runs all check functions in registry, returning a registry-like map where
  values are the result of evaluated check function for each key."
  [registry]
  (reduce-kv run-check! {} registry))

(s/fdef run-checks!
  :args (s/cat :registry ::registry)
  :ret  ::registry)

(defn healthy?
  "Returns true if x or xs are healthy. Returns false if no checks."
  [results]
  (if (empty? results) false (every? ::healthy? results)))

(s/fdef healthy?
  :args (s/cat :results sequential?)
  :ret  boolean?)

(defn run-all!
  "Runs checks which returns a lazy map of evaluated results. Realizes the
  check results and transforms the map into a sequence of results, mapping the
  check key into the result data."
  [registry]
  (let [assoc-key (fn [[k v]] (-> v (assoc ::key k)))
        ;; Note here that the usage of map returns a lazy sequence, which
        ;; should not be mixed with side-effects. But `run-checks!` uses
        ;; reduce-kv which, like reduce, is not lazy. So I think it's ok.
        results (map assoc-key (run-checks! registry))]
    {::healthy? (healthy? results)
     ::results results}))

(s/fdef run-all!
  :ret  ::all-results)

(defn result
  "Helper to construct a result."
  [healthy? detail]
  {::healthy? healthy? ::detail detail})

(defn healthy-result
  "Helper to construct a healthy result."
  [detail]
  (result true detail))

(defn unhealthy-result
  "Helper to construct a unhealthy result."
  [detail]
  (result false detail))
