(ns atlas.maps
  (:require [cljs.core.async :refer [chan put! close!]]
            [racehub.algebra :as a]
            [racehub.util :as u]
            [racehub.schema :refer [Channel]]
            [racehub.util.location :as l]
            [schema.core :as s :include-macros true])
  (:require-macros [cljs.core.async.macros :refer [go]]))

;; ## Google Schemas


(s/defschema LatLng "google.maps.LatLng" s/Any)
(s/defschema Map "google.maps.Map" s/Any)
(s/defschema Marker "google.maps.Marker" s/Any)

(s/defschema Autocomplete
  "google.maps.places.Autocomplete"
  s/Any)

(s/defn ranged :- [s/Str]
  [prefix :- s/Str top :- s/Int]
  (for [i (range top)]
    (str prefix (inc i))))

;; ### Place Schemas

(s/defschema PlaceType
  "Acceptable Place types are defined here:
   https://developers.google.com/places/documentation/supported_types"
  (apply s/enum (concat (ranged "administrative_area_level_" 3)
                        (ranged "sublocality_level_" 5)
                        ["subpremise" "colloquial_area" "country" "floor" "geocode" "intersection" "locality" "natural_feature"
                         "neighborhood" "political" "point_of_interest" "post_box" "postal_code" "postal_code_prefix" "postal_town"
                         "premise" "room" "route" "street_address" "street_number" "sublocality" "transit_station"])))

(s/defschema AddressComponent
  "https://developers.google.com/maps/documentation/javascript/reference#GeocoderAddressComponent"
  {:short_name s/Str
   :long_name s/Str
   :types [PlaceType]})

;; ## Utilities

(def default-zoom-level 15)

(def berkeley
  (google.maps.LatLng. 37.850459 -122.302233))

(def berkeley-coords
  {:latitude 37.850459
   :longitude -122.302233
   :zoom-level default-zoom-level})

(def default-map-opts
  "Default initial map options."
  {:zoom default-zoom-level
   :mapTypeId google.maps.MapTypeId.ROADMAP
   :center berkeley
   :styles [{:stylers [{:visibility "on"}]}]})

(s/defn marker :- Marker
  "Location is the location from the Browser's geoposition API"
  [position :- LatLng
   opts :- {s/Keyword s/Str}]
  (google.maps.Marker. (clj->js
                        (assoc opts :position position))))

(s/defn extract-position :- l/MarkerPosition
  [marker :- Marker]
  (let [position (.getPosition marker)]
    {:latitude (.lat position)
     :longitude (.lng position)
     :zoom-level (-> marker .getMap .getZoom)}))

(s/defn extract-google :- l/GoogleData
  "Accepts a PlaceResult, but the return value of .getPlace on the
  autocomplete bar doesn't actually have a type."
  [place-result]
  {:id (aget place-result "id")
   :reference (aget place-result "reference")
   :formatted-address (aget place-result "formatted_address")})

(s/defn expand-components :- {PlaceType [s/Str]}
  [components :- [AddressComponent]]
  (a/sum a/map-monoid
         (for [{:keys [types short_name]} components, type types]
           {type [short_name]})))

(s/defn extract-address :- l/Address
  "Accepts a PlaceResult, but the return value of .getPlace on the
  autocomplete bar doesn't actually have a type."
  [place-result]
  (let [components (-> (aget place-result "address_components")
                       (u/to-clj)
                       (expand-components))
        pick (comp first components)]
    (u/remove-values nil? {:street-number (pick "street_number")
                           :street-address (pick "route")
                           :city (or (pick "sublocality")
                                     (pick "locality"))
                           :zip (pick "postal_code")
                           :state (pick "administrative_area_level_1")
                           :country (pick "country")})))

(s/defn build-location :- l/Location
  "Accepts a google Marker item and a place result from the
   autocomplete bar and builds the appropriate schema entry for the
   regatta's location in the database."
  [marker :- Marker
   place-result]
  {:position (extract-position marker)
   :google (extract-google place-result)
   :address (extract-address place-result)})

;; ## Places API

(s/defn autocomplete :- Autocomplete
  "Optionally takes a set of AutocompleteOptions:
   https://developers.google.com/maps/documentation/javascript/reference#AutocompleteOptions"
  ([selector]
     (autocomplete selector #js {}))
  ([selector opts]
     (google.maps.places.Autocomplete. selector opts)))

(s/defn place-changed-chan :- Channel
  "Returns a channel that signals every time the place changes."
  ([ac :- Autocomplete]
     (place-changed-chan ac (chan)))
  ([ac :- Autocomplete output :- Channel]
     (let [f #(put! output (.getPlace ac))]
       (google.maps.event.addListener ac "place_changed" f)
       output)))

;; Markers API

(s/defn marker-event-chan :- Channel
  "Returns a channel that feeds out `zoom_changed` events from the
  supplied marker's associated Map instance.

  The channel also receives `place_changed` events from the marker;
  optionally, you can override that event by supplying a different
  event name in the `:event` key.

  Will return a new channel unless a `:channel` option is supplied."
  [opts :- {:marker Marker
            (s/optional-key :event) s/Str
            (s/optional-key :channel) Channel}]
  (let [output (:channel opts (chan))
        marker (:marker opts)
        f #(put! output (extract-position marker))
        event (:event opts "place_changed")]
    (when-let [map (.getMap marker)]
      (google.maps.event.addListener map "zoom_changed" f))
    (google.maps.event.addListener marker event f)
    output))

;; ## Maps API

(s/defn init-map :- Map
  "Installs an embedded map at the supplied dom element. Allowed
  options are described here:
  https://developers.google.com/maps/documentation/javascript/reference#MapOptions"
  ([elem]
     (init-map elem default-map-opts))
  ([elem opts :- {s/Any s/Any}]
     (google.maps.Map. elem (clj->js opts))))

(s/defn reposition!
  ([map :- Map position :- LatLng]
     (reposition! map position default-zoom-level))
  ([map :- Map
    position :- LatLng
    zoom :- l/ZoomLevel]
     (doto map
       (.setZoom zoom)
       (.panTo position))))

;; ## Athlete Helpers

(s/defn directions-link :- s/Str
  "Returns a URL that'll give directions to the race location in
  Google Maps."
  [location :- s/Str]
  (str "https://maps.google.com/maps?saddr=current+location&daddr="
  location))
