(ns atlas.map-field
  (:require [cljs.core.async :as a :refer [chan put!]]
            [om.core :as om]
            [om-bootstrap.input :as i]
            [om-bootstrap.util  :as u]
            [om-tools.core :refer-macros [defcomponentk]]
            [om-tools.dom :as d :include-macros true]
            [om-tools.schema :refer [cursor]]
            [atlas.maps :as m]
            [racehub.util.location :as l]
            [racehub.schema :refer [Channel]]
            [schema.core :as s :include-macros true])
  (:require-macros [cljs.core.async.macros :refer [go-loop]]))

;; ## Schema
(s/defschema MapView
  {:router Channel
   (s/optional-key :map-opts) {s/Any s/Any}
   (s/optional-key :position) l/MarkerPosition
   (s/optional-key :place-update-chan) Channel})

;; ## Functions

(s/defn place-marker!
  "Forces the position of the marker in the application state to the
  supplied position."
  [state router position :- m/LatLng]
  (swap! state (fn [m]
                 (let [marker (if-let [marker (:marker m)]
                                (doto marker (.setPosition position))
                                (let [opts {:map (:map m), :draggable true}
                                      new-marker (m/marker position opts)]
                                  (m/marker-event-chan
                                   {:marker new-marker
                                    :event "dragend"
                                    :channel router})
                                  new-marker))]
                   (assoc m :marker marker)))))

(s/defn relocate!
  "Takes a state atom with :map and :router keys and places a marker
  at the supplied initial location.

  The first time the marker is place, relocate! starts sending any
  change to the marker into the supplied :router channel. These
  updates occur when the containing map changes its zoom level, or
  when the marker finishes getting dragged."
  [state location :- l/Location]
  (let [{:keys [router map]} @state
        point (google.maps.LatLng. (:latitude location)
                                   (:longitude location))]
    (place-marker! state router point)
    (m/reposition! map point (:zoom-level location))))

(s/defn event-loop
  [state
   router :- Channel
   position :- (cursor l/MarkerPosition)
   place-update-chan :- Channel]
  (go-loop []
    (let [new-place (<! place-update-chan)
          geometry (.-geometry new-place)
          loc (.-location geometry)]
      (relocate! state {:latitude (.lat loc)
                        :longitude (.lng loc)
                        :zoom-level (.getZoom (:map @state))}))
    (recur)))

;; ## Map Component

(defcomponentk google-map*
  "The component holds a Google Map. The component drops a marker on
  initialization at the supplied position, then starts ignoring the
  position in favor of the internal Marker state.

  All position updates to the marker report out via the
  supplied :router channel."
  [[:data {map-opts m/default-map-opts} position children]
   owner state]
  (init-state [_] {:map nil})
  (will-mount
   [_] (when-let [ch (:place-update-chan @state)]
         (event-loop state (:router @state) position ch)))
  (did-mount [_] (let [map-div (om/get-node owner "map_canvas")
                       map (m/init-map map-div map-opts)]
                   (swap! state assoc :map map)
                   (relocate! state position)))
  (render [_]
          (d/div
           {:class "form-group input-field"}
           (d/div
            {:class "col-sm-9 col-sm-offset-3"}
            (d/div {:id "map_canvas"
                    :ref "map_canvas"})
            children))))

(s/defn google-map
  "Generates a Google Map component. The :router field gets a
  l/Position item every time the marker moves or changes."
  [opts :- MapView & children]
  (->google-map*
   (-> (dissoc opts :place-update-chan :router)
       (update-in [:map-opts] #(merge m/default-map-opts %))
       (assoc :children children))
   {:init-state {:place-update-chan (:place-update-chan opts)
                 :router (a/map> (fn [x] [:position x]) (:router opts))}}))

;; ## Location Field

(defcomponentk location-field
  "The Location field uses Google Places autocomplete
   internally. Autocomplete has some issues with onChange and onBlur;
   after clicking one of the options in the dropdown (instead of
   tabbing away), the field won't update until after onBlur has
   fired.

    This component gets around that issue by waiting until the
   `place_changed` event is fired by the Autocomplete widget, and only
   then reading the value of the original input component.

   :init-state options:

   :router (channel, required)
   :place-update-chan (channel, optional)

   If you supply a :place-update-chan, every time the place changes an
   instance of google.maps.places.PlaceResult will drop into the
   channel."
  [data owner state]
  (did-mount [_]
             (let [input (om/get-node owner "input")
                   ac (m/autocomplete input #js {:types #js ["geocode"]})
                   c (m/place-changed-chan ac)
                   router (:router @state)]
               (go-loop []
                 (let [place (a/<! c)
                       loc {:google (m/extract-google place)
                            :address (m/extract-address place)}]
                   (a/>! router [:location (.-value input)])
                   (a/>! router [:loc loc]))
                 (recur))
               (when-let [update-ch (:place-update-chan @state)]
                 (m/place-changed-chan ac update-ch))))
  (render-state [_ {:keys [router]}]
                (let [handler #(put! router [:location (.. % -target -value)])
                      props {:type "text"
                             :on-blur handler}]
                  (i/input
                   (u/merge-props props data)))))

;; ## Combined Map Field

(defcomponentk map-field
  "This component brings together the location field and the map
  field. They're linked internally by a channel, so whenever a new
  location is entered into the autocomplete box, the map centers on
  that new component.

  The supplied position is the initial position of the map and
  marker."
  [[:data location position errors]]
  (init-state [_] {:place-update-chan (chan)})
  (render-state
   [_ {:keys [router place-update-chan]}]
   (d/div
    (->location-field
     {:label "Location:"
      :addon-before (i/glyph "map-marker")
      :default-value location
      :help (first errors)
      :bs-style (when errors "error")
      :placeholder "Enter the race venue's location."
      :label-classname "col-sm-3"
      :wrapper-classname "col-sm-9"}
     {:init-state {:router router
                   :place-update-chan place-update-chan}})
    (google-map
     {:position position
      :router router
      :place-update-chan place-update-chan}
     (d/div
      {:class "help-block"}
      "Adjust the marker on the map to point to the start of the race course.")))))
