(ns atlas.geolocation
  (:require [cljs.core.async :refer [chan put! close! <!]]
            [racehub.util :as u]
            [racehub.schema :refer [Channel PositiveInt]]
            [schema.core :as s :include-macros true])
  (:require-macros [cljs.core.async.macros :refer [go]]))

;; Great docs for this stuff at
;; https://developer.mozilla.org/en-US/docs/Web/API/Geolocation.clearWatch

;; ## Schemas

(s/defschema LatLng
  "google.maps.LatLng"
  s/Any)

(def failure-map
  {1 :permission-denied
   2 :position-unavailable
   3 :timeout})

(s/defschema GeoFailure
  (apply s/enum (vals failure-map)))

(s/defschema PositionResult
  (s/either {:success (s/eq true)
             :position LatLng}
            {:success (s/eq false)
             :failure {:message s/Str
                       :reason GeoFailure}}))

(s/defschema WatchID
  (s/named s/Int "ID of the watch."))

(s/defschema PositionUpdates
  "Send anything down the kill channel to shut down the position
  updates."
  {:kill Channel
   :channel Channel})

;; More info here: https://developer.mozilla.org/en-US/docs/Web/API/PositionOptions
(s/defschema Options
  {(s/optional-key :enableHighAccuracy) (s/named s/Bool "Tells the device to use GPS or wifi to get a better signal. Might cost more power.")
   (s/optional-key :timeout) (s/named PositiveInt "Max length of time (in milliseconds) the device is allowed to take in order to return a position.")
   (s/optional-key :maximumAge) (s/named PositiveInt "The amount of time in millis that the device is allowed to cache positions.")})

;; ## Browser Geoposition API

(s/defn position->latlng :- LatLng
  "Accepts a geoposition."
  [position]
  (google.maps.LatLng. (.. position -coords -latitude)
                       (.. position -coords -longitude)))

(s/defn report-position! :- s/Bool
  [c :- Channel]
  (fn [position]
    (put! c {:success true
             :position (position->latlng position)})))

(s/defn report-failure! :- s/Bool
  [c :- Channel]
  (fn [error]
    (put! c {:success false
             :error error})))

(s/defn clear-watch
  "Destroys the supplied watch in the browser's geolocation
  interface."
  [id :- s/Int]
  (.clearWatch (.-geolocation js/navigator) id))

(s/defn position-chan :- PositionUpdates
  "Returns a channel that sends periodic updates on the browser's
  position AND a kill channel. Send anything down the kill channel to
  shut down updates."
  ([] (position-chan {}))
  ([opts :- Options]
     (let [output (chan)
           kill (chan)
           watch-id (.watchPosition (.-geolocation js/navigator)
                                    (report-position! output)
                                    (report-failure! output)
                                    (clj->js (or opts {})))]
       (go (<! kill)
           (clear-watch watch-id)
           (close! output))
       {:kill kill
        :channel output})))

(s/defn current-position-chan :- Channel
  "Returns a channel that reports the current user's position (or a
  failure!), then shuts down."
  ([] (current-position-chan {}))
  ([opts :- Options]
     (u/with-chan
       (fn [output]
         (let [with-close (fn [res] (close! output) res)]
           (.getCurrentPosition (.-geolocation js/navigator)
                                (comp with-close (report-position! output))
                                (comp with-close (report-failure! output))
                                (clj->js (or opts {}))))))))
