(ns quantum.core.compare
  (:refer-clojure :exclude
    [= not= < > <= >= max min max-key min-key neg? pos? zero? - -' + inc compare
     reduce, transduce, first])
  (:require
    [clojure.core                    :as core]
    [quantum.core.log                :as log
      :refer [prl!]]
    [quantum.core.collections.core   :as ccoll
      :refer [conj?! ?persistent! ?transient!, first, join]]
    [quantum.core.compare.core       :as ccomp]
    [quantum.core.convert.primitive  :as pconv
      :refer [->boxed ->boolean ->long]]
    [quantum.core.error :as err
      :refer [TODO]]
    [quantum.core.fn                 :as fn
      :refer [fn&2 rfn aritoid]]
    [quantum.core.macros
      :refer [defnt #?@(:clj [defnt' variadic-proxy variadic-predicate-proxy])]]
    [quantum.core.numeric.convert
      :refer [->num ->num&]]
    [quantum.core.numeric.operators  :as op
      :refer [- -' + abs inc div:natural]]
    [quantum.core.numeric.predicates :as pred
      :refer [neg? pos? zero?]]
    [quantum.core.numeric.types      :as ntypes]
    [quantum.core.reducers           :as red
      :refer [reduce, transduce]]
    [quantum.core.vars
      :refer [defalias defaliases]])
#?(:cljs (:require-macros
    [quantum.core.compare            :as self
      :refer [< > <= >=]]))
#?(:clj
  (:import
    clojure.lang.BigInt quantum.core.Numeric)))

(defaliases ccomp
  compare
  min-key first-min-key second-min-key
  max-key first-max-key second-max-key
  #?@(:clj [compare-1d-arrays-lexicographically
            =   =&   not=     not=&
            <   <&   comp<    comp<&
            <=  <=&  comp<=   comp<=&
            >   >&   comp>    comp>&
            >=  >=&  comp>=   comp>=&
            min min& comp-min comp-min&
            max max& comp-max comp-max&
            min-byte   max-byte
            min-char   max-char
            min-short  max-short
            min-int    max-int
            min-long   max-long
            min-float  max-float
            min-double max-double
            #_first-min #_second-min
            #_first-max #_second-max]))

(defn gen-comp-keys-into:rf
  ; TODO use `reduce-multi` or `multiplex`
  ([initf compf kf]
    (let [rf (rfn [[ret best] x]
               (if (identical? best red/sentinel)
                   [(conj?! ret x) x]
                   (let [vret (kf best) vx (kf x)]
                     (cond (=     vret vx) [(conj?! ret x)     x]
                           (compf vret vx) [ret                best]
                           :else           [(conj?! (initf) x) x]))))]
      (aritoid (fn [] [(?transient! (initf)) red/sentinel])
               (fn [[ret _]] (?persistent! ret))
               rf
               rf))))

(defn comp-keys-into
  "Like `max-key`, but returns a collection of all equally 'max' values (even
   when no tie is present), generated by `initf`."
  ([initf compf kf] nil) ; TODO really, the min of whatever it is; maybe gen via `(kf)` ?
  ([initf compf kf x] x)
  ([initf compf kf x y] ; TODO merge this logic with `gen-comp-keys-into:rf`
    (let [vx (kf x) vy (kf y)]
      (cond (=     vx vy) (-> (initf) (conj?! x) (conj?! y)) ; TODO (conj?! (initf) x y)
            (compf vx vy) (conj?! (initf) x)
            :else         (conj?! (initf) y))))
  ([initf compf kf x y & more]
    (transduce (gen-comp-keys-into:rf initf compf kf) (conj more y x))))

(defn max-keys
  "Like `max-key`, but returns a collection of all equally 'max' values (even
   when no tie is present)."
  ([kf] nil) ; TODO really, the min of whatever it is; maybe gen via `(kf)` ?
  ([kf x] x)
  ([kf x y       ]       (comp-keys-into vector core/> kf x y))       ; TODO use this/>
  ([kf x y & more] (apply comp-keys-into vector core/> kf x y more))) ; TODO use this/>

(defn reduce-first-max-key  [kf xs] (red/reduce-sentinel (fn&2 first-max-key  kf) xs))
(defn reduce-second-max-key [kf xs] (red/reduce-sentinel (fn&2 second-max-key kf) xs))
(defn reduce-max-key        [kf xs] (red/reduce-sentinel (fn&2 max-key        kf) xs))

(defn reduce-first-min-key  [kf xs] (red/reduce-sentinel (fn&2 first-min-key  kf) xs))
(defn reduce-second-min-key [kf xs] (red/reduce-sentinel (fn&2 second-min-key kf) xs))
(defn reduce-min-key        [kf xs] (red/reduce-sentinel (fn&2 min-key        kf) xs))

(defn reduce-first-min      [   xs] (red/reduce-sentinel ccomp/first-min-temp           xs))
(defn reduce-second-min     [   xs] (red/reduce-sentinel ccomp/second-min-temp          xs))
(defn reduce-min            [   xs] (red/reduce-sentinel ccomp/min-temp                 xs))

(defn reduce-first-max      [   xs] (red/reduce-sentinel ccomp/first-max-temp           xs))
(defn reduce-second-max     [   xs] (red/reduce-sentinel ccomp/second-max-temp          xs))
(defn reduce-max            [   xs] (red/reduce-sentinel ccomp/max-temp                 xs))

(defn reduce-maxes          [               xs] (transduce (gen-comp-keys-into:rf vector core/> identity) xs)) ; TODO use this/>
(defn reduce-max-keys       [            kf xs] (transduce (gen-comp-keys-into:rf vector core/> kf      ) xs)) ; TODO use this/>
(defn reduce-max-keys-into  [initf       kf xs] (transduce (gen-comp-keys-into:rf initf  core/> kf      ) xs)) ; TODO use this/>
(defn reduce-comp-keys      [      compf kf xs] (transduce (gen-comp-keys-into:rf vector compf  kf      ) xs)) ; TODO use this/>
(defn reduce-comp-keys-into [initf compf kf xs] (transduce (gen-comp-keys-into:rf initf  compf  kf      ) xs)) ; TODO use this/>

(defn rcompare
  "Reverse comparator."
  {:attribution "taoensso.encore, possibly via weavejester.medley"}
  [x y] (compare y x))

(defn greatest
  "Returns the 'greatest' element in coll in O(n) time."
  {:attribution "taoensso.encore, possibly via weavejester.medley"}
  [coll & [?comparator]]
  (let [comparator (or ?comparator rcompare)]
    (reduce
      (fn ([] nil) ([a b] (if (pos? (comparator a b)) b a)))
      coll)))

(defn least
  "Returns the 'least' element in coll in O(n) time."
  {:attribution "taoensso.encore, possibly via weavejester.medley"}
  [coll & [?comparator]]
  (let [comparator (or ?comparator rcompare)]
    (reduce
      (fn ([] nil) ([a b] (if (neg? (comparator a b)) b a)))
      coll)))

(defn unsorted-by
  "Returns which elements are unsorted, as by `kf` and `comparef`."
  {:example '{(unsorted-by identity core/< [0 1 2 3 6 3 7])
              [[5 3]]}}
  ([kf xs] (unsorted-by kf core/compare xs))
  ([kf comparef xs]
    (let [xs'      (transient [])
          comparef (ccomp/fn->comparator comparef)]
      (red/reducei-sentinel
        (fn [a b i]
          (when-not (neg? (#?@(:clj  [.compare ^java.util.Comparator comparef]
                         :cljs [comparef])
                     (kf a) (kf b)))
            (conj! xs' [i b]))
          b) xs)
      (persistent! xs'))))

(defn unsorted
  "Returns which elements are unsorted, as by `comparef`."
  {:example '{(unsorted core/< [0 1 2 3 6 3 7])
              [[5 3]]}}
  ([xs] (unsorted-by identity xs))
  ([comparef xs] (unsorted-by identity comparef xs)))

#?(:clj
(defmacro min-or [a b else]
 `(let [a# ~a b# ~b]
    (cond (< a# b#) a#
          (< b# a#) b#
          :else ~else))))

#?(:clj
(defmacro max-or [a b else]
 `(let [a# ~a b# ~b]
    (cond (> a# b#) a#
          (> b# a#) b#
          :else ~else))))

#_(defn extreme-comparator [comparator-n]
  (get {> num/greatest
        < num/least   }
    comparator-n))

; ===== APPROXIMATION ===== ;

(defn approx=
  "Return true if the absolute value of the difference between x and y
   is less than eps."
  {:todo #{"`core/<` -> `<` "}}
  [x y eps]
  (core/< (abs (- x y)) eps))

(defn within-tolerance?
  {:todo #{"`core/<` -> `<` "}}
  [n total tolerance]
  (and (core/>= n (- total tolerance))
       (core/<= n (+ total tolerance))))

(defnt' within-percent-tolerance? [^number? actual ^number? expected percent-tolerance]
  (core/< (div:natural (double (abs (core/- expected actual))) expected) ; TODO remove `double` cast!!!
          percent-tolerance))
