(ns quantum.core.data.bytes
  (:refer-clojure :exclude [reverse]))

; SRC ns mikera.cljutils.bytes

(require
  '[quantum.core.string           :as str]
  '[quantum.core.collections.core         :refer :all]
  '[quantum.core.data.binary      :as bin :refer :all]
  '[quantum.core.logic                    :refer :all]
  '[quantum.core.data.array       :as arr :refer [byte-array+ aset!]])
(import  'java.util.Arrays)

(set! *warn-on-reflection* true)
(set! *unchecked-math* true)

(def BYTE-ARRAY-CLASS (Class/forName "[B"))

(defn reverse 
  (^bytes [^bytes bs]
    (let [n (alength bs)
          res (byte-array n)]
      (dotimes [i n]
        (aset res i (aget bs (- n (inc i)))))
      res)))

(defn join 
  "Concatenates two byte arrays"
  (^bytes [^bytes a ^bytes b]
    (let [al (int (alength a))
          bl (int (alength b))
          n (int (+ al bl))
          ^bytes res (byte-array n)]
      (System/arraycopy a (int 0) res (int 0) al)
      (System/arraycopy b (int 0) res (int al) bl)
      res)))

(defn slice
  "Slices a byte array with a given start and length"
  (^bytes [a start]
    (slice a start (- (alength ^bytes a) start)))
  (^bytes [a start length]
    (let [al (int (alength ^bytes a))
          ^bytes res (byte-array length)]
      (System/arraycopy a (int start) res (int 0) length)
      res)))

; (defn to-hex-string 
;   "Converts a byte array to a string representation , with space as a default separator."
;   ([^bytes bs]
;     (to-hex-string bs " "))
;   ([^bytes bs separator]
;     (str/join separator (map #(hex/hex-string-from-byte %) bs))))

(defn unchecked-byte-array 
  "Like clojure.core/byte-array but performs unchecked casts on sequence values."
  (^bytes [size-or-seq] 
    (. clojure.lang.Numbers byte_array 
      (if (number? size-or-seq) 
        size-or-seq
        (map unchecked-byte size-or-seq ))))
  (^bytes [size init-val-or-seq] 
    (. clojure.lang.Numbers byte_array size 
      (if (sequential? init-val-or-seq) 
        (map unchecked-byte init-val-or-seq )
        init-val-or-seq))))

(defn bytes=
  "Compares two byte arrays for equality."
  ([^bytes a ^bytes b]
    (Arrays/equals a b)))

(defn ^String bytes-to-hex
  "Convert a byte array to a hex string."
  [^"[B" digested]
  (let [^"[C" hex-arr   (.toCharArray "0123456789abcdef")
        ^"[C" hex-chars (-> digested count (* 2) char-array)]
    (loop [i 0] 
      (if (< i (count digested))
          (let [v           (-> digested (get+ i) (bit-and 0xFF))
                bit-shifted (-> hex-arr  (get+ (>>>     v 4   )))
                bit-anded   (-> hex-arr  (get+ (bit-and v 0x0F)))]
                (aset hex-chars (* i 2)       bit-shifted)
                (aset hex-chars (+ (* i 2) 1) bit-anded)
              (recur (inc i)))))
    (String. hex-chars)))

(defn ^"[B" str->cstring [^String s]
  (when (nnil? s)
    (let [^"[B" bytes  (.getBytes s)
          ^"[B" result (byte-array+ (-> bytes count inc))]
        (System/arraycopy
          bytes  0
          result 0
          (count bytes))
        (aset! result (-> result count dec) (byte 0))
        result)))

