(ns pulley.store.mysql
  (:require [clojure.java.jdbc     :as jdbc]
            [pulley.store          :as store]))

;;--------------------------------------------------------------------
;; schema

(defn- table-exists?
  [db table-name]
  (-> (jdbc/query db
                  ["SHOW TABLES LIKE ?" table-name])
      empty?
      not))

(defn- create-table
  [db table-name]
  (jdbc/execute! db
    [(str "CREATE TABLE `" table-name "` (
             `id` int(20) NOT NULL AUTO_INCREMENT,
             `timestamp` int(11) NOT NULL,
             `data` blob NOT NULL,
             PRIMARY KEY (`id`),
             KEY `timestamp` (`timestamp`)
           ) ENGINE=InnoDB DEFAULT CHARSET=latin1;")]))

(defn- ensure-schema
  [db table-name]
  (when-not (table-exists? db table-name)
    (create-table db table-name)))

;;--------------------------------------------------------------------
;; sql statements

(defn- insert-event
  [db table-name event-data]
  (let [res (jdbc/insert! db table-name event-data)]
    (-> res :generated_key)))

(defn- events-from
  [db table-name id count]
  (-> (jdbc/query db
                  [(str "SELECT id, timestamp, data
           FROM `" table-name "`
           WHERE id >= ? AND id < ?")
                   id (+ id count)]
                  )
      vec))

(defn- timestamp->id
  [db table-name timestamp]
  (-> (jdbc/query db
                  [(str "SELECT MIN(id) AS `id` FROM `" table-name "`"
                        "WHERE timestamp >= ?")
                   timestamp])
      first
      :id))

(defn event-store
  [db table-name]
  (reify store/EventStore
    (-write
      [store event]
      (insert-event db table-name event))
    (-events-from
      [store id count]
      (events-from db table-name id count))
    (-timestamp->id
      [store timestamp]
      (timestamp->id db table-name timestamp))))

(defn make-event-store
  ([db table-name]
     (make-event-store db table-name {}))
  ([db table-name opts]
     (when (:ensure-schema opts)
       (ensure-schema db table-name))
     (event-store db table-name)))

(comment

  (def mysql-db {:subprotocol "mysql"
                 :subname "//127.0.0.1:3306/events_test"
                 :user "root"})

  

  )
