(ns pulley.store
  (:require [clj-time.core   :as time]
            [clj-time.coerce :as time-coerce]
            [clojure.edn]))

;;--------------------------------------------------------------------
;; serialization

(def ^:private utf8 (java.nio.charset.Charset/forName "UTF-8"))

(defn- edn->bytes
  [obj]
  (.getBytes (pr-str obj) utf8))

(defn- bytes->edn
  [bytes]
  (clojure.edn/read-string (String. bytes utf8)))

;;--------------------------------------------------------------------
;; helpers

(defn- msecs->secs
  [msecs]
  (/ msecs 1000))

(defn unix-time
  "Return the current unix time in seconds."
  []
  (-> (System/currentTimeMillis) msecs->secs long))


;;--------------------------------------------------------------------
;; event store

(defprotocol EventStore
  "An EventStore provides event persistence and very simple querying
  capabilities. An event is any edn object, but typically a map."

  (-write
    [store event-data]
    "Persist the given event-data, returning its assigned `id`. Event
    data is a map with the following keys:

      :timestamp   - unix time in seconds
      :data        - event data as a byte array ")

  (-events-from
    [store id count]
    "Retrieve up to `count` events, starting from `id`.")

  (-timestamp->id
    [store timestamp]
    "Given a unix time in seconds, return the earliest event `id` that
    occured at that time."))

;; public API

(defn write
  [store event]
  (-write store {:timestamp (unix-time)
                 :data      (edn->bytes event)}))

(defn events-from
  [store id count]
  (mapv (fn [event]
          (update-in event [:data] bytes->edn))
        (-events-from store id count)))

(defn timestamp->id
  [store timestamp]
  (-timestamp->id store timestamp))


(defn resolve-timestamp
  "Given a unix time in seconds, return the earliest event `id` that
  occurred at that time."
  [store timestamp]
  {:pre [(satisfies? EventStore store)
         (integer? timestamp)]}
  (timestamp->id store timestamp))

(defn resolve-date-time
  "Given a date-time, or something coercable to a date-time, return
  the earliest event `id` that occurred at that time."
  [store date-time]
  {:pre [(satisfies? EventStore store)]}
  (resolve-timestamp store (msecs->secs (time-coerce/to-long date-time))))

