(ns pulley.ring
  (:require [pulley.store                   :as store]
            [ring.util.response             :as response]
            [ring.middleware.params         :as ring-params]
            [ring.middleware.keyword-params :as ring-keyword-params]
            [clojure.string                 :as string]
            [clj-time.format                :as time-format]
            [cemerick.url                   :as url])
  (:import [java.net URL]))



(defn- parse-long
  [v]
  (when v
    (try
      (Long/parseLong v)
      (catch NumberFormatException e))))

(defn- original-server-name
  [{:keys [server-name] :as req}]
  (if-let [host (get-in req [:headers "host"])]
    host
    server-name))

(defn- feed-url
  [{:keys [scheme headers server-name server-port uri] :as req} from]
  (str (name scheme)
       "://"
       (str (original-server-name req) ":" server-port)
       uri
       "?from=" from))

(defn- full-page?
  [page-size events]
  (= page-size (count events)))

(defn- etag
  [from events]
  (if (< 0 (count events))
    (str from "-" (+ from (count events)))
    (str from)))

(defn- response-map
  [req from page-size events]
  (cond-> {}

          (not (empty? events))
          (assoc :events (vec events))

          (< 0 (count events))
          (assoc :next (feed-url req (+ from (count events))))))


(defn- response-headers
  [req from page-size events]
  (cond-> {"Content-Type" "application/edn;charset=UTF8"}

          (< 0 (count events))
          (assoc "ETag" (etag from events))

          (full-page? page-size events)
          (assoc "Cache-Control" "max-age=31536000")))

(defn- events-response
  [store page-size req]
  (let [from   (-> req :params :from parse-long (or 1) (max 1))
        events (store/events-from store
                                  from
                                  page-size)]
    {:status  200
     :headers (response-headers req from page-size events)
     :body    (pr-str (response-map req from page-size events))}))

(defn- timestamp-response
  [store req]
  (let [unix-time (-> req :params :from-timestamp parse-long (or 0) (max 1))
        id        (store/resolve-timestamp store unix-time)]
    (if id
      {:status 302
       :headers {"Location" (feed-url req id)}}
      {:status 404})))

(defn- date-time-response
  [store req]
  (let [date-time (->> req
                       :params
                       :from-date-time
                       time-format/parse)
        id        (store/resolve-date-time store date-time)]
    (if id
      {:status 302
       :headers {"Location" (feed-url req id)}}
      {:status 404})))

(defn- make-handler
  [store page-size]
  (fn [req]
    (cond
     (-> req :params :from)           (events-response store page-size req)
     (-> req :params :from-timestamp) (timestamp-response store req)
     (-> req :params :from-date-time) (date-time-response store req)
     :default                         (events-response store page-size req))))

(defn handler
  [store page-size]
  (-> (make-handler store page-size)
      
      (ring-keyword-params/wrap-keyword-params)
      (ring-params/wrap-params)))




