(ns pulley.ring
  (:require [pulley.store                   :as store]
            [ring.util.response             :as response]
            [ring.middleware.params         :as ring-params]
            [ring.middleware.keyword-params :as ring-keyword-params]
            [clj-time.format                :as time-format]
            [cemerick.url                   :as url])
  (:import [java.net URL]))


;; less important, but still
"application/vnd.pulley+edn"


;;--------------------------------------------------------------------
;; 55 events (1-55)

"/"
'{:first "/?from=1"
  :last  "/?from=55"}
;; ETag: 1-55


"/?from=1"
'{:next  "/?from=11"  ;; from + page-size
  :data  [..........]}
;; ETag: 1-10

"/?from=11"
'{:next  "/?from=21"  ;; from + page-size
  :prev  "/?from=1"   ;; 
  :data  [..........]}
;; ETag: 11-20

"/?from=51"
'{:next "?from=55"
  :prev "?from=41"
  :data [....]}
;; ETag: 51-54

"/?from=55"
'{:prev "?from="
  :data nil}
;; ETag: 55

;; If-None-Match: 55
;; 304
;; ETag: 55

;; If-None-Match: 55
;; 304
;; ETag: 55

;; If-None-Match: 55
"/?from=55"
'{:next "?from=56"
  :prev "?from="
  :data [.]}

;; If-None-Match: 55-56


;;--------------------------------------------------------------------
;; page-size re-alignment?

"/?from=7"
'{:next  "/?from=11"  ;; from + page-size
  :prev  "/?from=1"   ;; 
  :data  [...]}       ;; only three things this time
;; ETag: 7-10


;;--------------------------------------------------------------------
;; timestamp resolution

"/?from-timestamp=12345"
;; resolve id and redirect to /?from=34334

"/?from-date-time=2013-04-05"

(defn- parse-long
  [v]
  (when v
    (try
      (Long/parseLong v)
      (catch NumberFormatException e))))

(defn- feed-url
  [{:keys [scheme headers server-name server-port uri]} from]
  (str (name scheme)
       "://"
       (get "host" headers (str server-name ":" server-port))
       uri
       "?from=" from))

(defn- full-page?
  [page-size events]
  (= page-size (count events)))

(defn- etag
  [from events]
  (if (< 0 (count events))
    (str from "-" (+ from (count events)))
    (str from)))

(defn- response-map
  [req from page-size events]
  (cond-> {}

          (not (empty? events))
          (assoc :events (vec events))

          (< 0 (count events))
          (assoc :next (feed-url req (+ from (count events))))))


(defn- response-headers
  [req from page-size events]
  (cond-> {"Content-Type" "application/edn;charset=UTF8"}

          (< 0 (count events))
          (assoc "ETag" (etag from events))

          (full-page? page-size events)
          (assoc "Cache-Control" "max-age=31536000")))

(defn- events-response
  [store page-size req]
  (let [from   (-> req :params :from parse-long (or 1) (max 1))
        events (store/events-from store
                                  from
                                  page-size)]
    {:status  200
     :headers (response-headers req from page-size events)
     :body    (pr-str (response-map req from page-size events))}))

(defn- timestamp-response
  [store req]
  (let [unix-time (-> req :params :from-timestamp parse-long (or 0) (max 1))
        id        (store/resolve-timestamp store unix-time)]
    (if id
      {:status 302
       :headers {"Location" (feed-url req id)}}
      {:status 404})))

(defn- date-time-response
  [store req]
  (let [date-time (->> req
                       :params
                       :from-date-time
                       time-format/parse)
        id        (store/resolve-date-time store date-time)]
    (if id
      {:status 302
       :headers {"Location" (feed-url req id)}}
      {:status 404})))

(defn- make-handler
  [store page-size]
  (fn [req]
    (cond
     (-> req :params :from)           (events-response store page-size req)
     (-> req :params :from-timestamp) (timestamp-response store req)
     (-> req :params :from-date-time) (date-time-response store req)
     :default                         (events-response store page-size req))))

(defn handler
  [store page-size]
  (-> (make-handler store page-size)
      
      (ring-keyword-params/wrap-keyword-params)
      (ring-params/wrap-params)))




