(ns pulley.store
  (:require [clj-time.core   :as time]
            [clj-time.coerce :as time-coerce]))

(defprotocol EventStore
  "An EventStore provides event persistence and very simple querying
  capabilities. An event is any edn object, but typically a map."

  (write
    [store event]
    "Persist the given event, returning its assigned `id`.")

  (events-from
    [store id count]
    "Retrieve up to `count` events, starting from `id`.")

  (timestamp->id
    [store timestamp]
    "Given a unix time in seconds, return the earliest event `id` that
    occured at that time."))

(defn- msecs->secs
  [msecs]
  (/ msecs 1000))

(defn unix-time
  "Return the current unix time in seconds."
  []
  (-> (System/currentTimeMillis) msecs->secs long))

(defn resolve-timestamp
  "Given a unix time in seconds, return the earliest event `id` that
  occurred at that time."
  [store timestamp]
  {:pre [(satisfies? EventStore store)
         (integer? timestamp)]}
  (timestamp->id store timestamp))

(defn resolve-date-time
  "Given a date-time, or something coercable to a date-time, return
  the earliest event `id` that occurred at that time."
  [store date-time]
  {:pre [(satisfies? EventStore store)]}
  (resolve-timestamp store (msecs->secs (time-coerce/to-long date-time))))

