(ns pulley.state-process)

(defprotocol StateProcess
  (stop [this])
  (current-state [this]))

(defn state-process
  "Starts a state process in a different thread and returns an object
  implementing the StateProcess protocol. The process is initiated with
  initial-state and continuesly brings the state to a new point by applying
  evolve to the current state.

  The returned StateProcess object has two main methods:

   stop  for stopping the process by not evolving the state the next time
         it would happen, and
   current-state  returns the current state.

  The returned object can also be dereferenced. This will join the current
  thread with the process state and return a map with the keys :process-state
  (the future performing the steps) and stopped-state, the last state the
  process was in."
  [evolve initial-state]
  (let [running? (atom true)
        state (atom initial-state)
        process (future
                  (while @running?
                    (swap! state evolve)))]
    (reify
      StateProcess
      (stop [this] (reset! running? false))
      (current-state [this] @state)
      clojure.lang.IDeref
      (deref [this]
        (let [process-state (try @process (catch Throwable t t))]
          {:process-state process-state
           :stopped-state @state})))))
