(ns pulley.consume
  (:require [pulley.state-process  :as state-process]
            [clojure.edn]
            [clj-http.client       :as http]
            [clj-time.core         :as time]
            [clj-time.coerce       :as time-coerce]
            [clj-time.format       :as time-format]
            [cemerick.url          :as url]))

(defn- fetch
  [feed-url http-client-opts]
  (let [res (http/get feed-url (assoc-in
                                http-client-opts
                                [:headers "Accept"]
                                "application/edn"))]
    (if (http/success? res)
      (clojure.edn/read-string (:body res))
      nil)))


;;--------------------------------------------------------------------
;; url helpers
;; TODO: Maybe shared?

(defn- from-url
  [feed-url from]
  (-> (url/url feed-url)
      (assoc-in [:query "from"] from)))

(defn- from-timestamp-url
  [feed-url timestamp]
  (-> (url/url feed-url)
      (assoc-in [:query "from-timestamp"] timestamp)))

(defn- from-date-time-url
  [feed-url date-time]
  (-> (url/url feed-url)
      (assoc-in [:query "from-date-time"]
                (->> date-time
                     (time-coerce/to-date-time)
                     (time-format/unparse (time-format/formatters :date-time-no-ms))))))


;;--------------------------------------------------------------------
;; consuming
;;

(defn- start-url
  [feed-url opts]
  (let [{:keys [start-id start-timestamp start-date-time]} opts]
    (cond
     start-id        (from-url feed-url start-id)
     start-timestamp (from-timestamp-url feed-url start-timestamp)
     start-date-time (from-date-time-url feed-url start-date-time))))

(defmulti consume-state :state)

(defmethod consume-state :wait
  [state]
  (println "Waiting")
  (Thread/sleep 5000)
  (assoc state :state :walk))

(defmethod consume-state :walk
  [state]
  (println "Walking")
  (try
    (let [next-state
          (let [res (fetch (:url state) (:http-client-opts state))
                handle-event (:handle-event state)]
            (when res
              (doseq [event (:events res)]
                (handle-event event))
              (if-let [next (:next res)]
                (assoc state :state :walk :url next)
                (assoc state :state :wait))))]
      (Thread/sleep 10)
      next-state)
    (catch Exception e
      (assoc state
        :state :error
        :exception e))))

(defmethod consume-state :error
  [state]
  (println "Error!!!")
  (println (:exception state))
  (Thread/sleep 10000)
  (-> state
      (assoc :state :walk)
      (dissoc :exception)))

(defn make-consumer
  "Creates a consumer thread which will poll the events available from feed-url
  continuously. The process will wait 10ms between requests, and when there is
  no more data available, it will pause for 5s before continuing. If an error
  http code is returned from the server, the process will wait 10s before trying
  again. handle-event will be invoked on every event returned by the feed. opts
  should contain one of the keys :start-id, :start-timestamp or start-date-time.

  If special parameters are needed for the clj-http library, these can be
  supplied uner the :http-client-opts key in the opts map.

  The returned object can be stopped by invoking stop. The thread it starts can
  be joined by dereferencing the object."
  [feed-url handle-event {:keys [http-client-opts] :as opts}]
  (state-process/state-process
   consume-state
   {:state :walk
    :url (str (start-url feed-url opts))
    :handle-event handle-event
    :http-client-opts http-client-opts}))

(def stop pulley.state-process/stop)

(comment

  (fetch "http://localhost:8001/?from=2")
  (fetch "http://localhost:8001/?from-timestamp=1368726132")
  (fetch "http://localhost:8001/?from-date-time=2013-05-17T11:42")

  (def consumer
    (make-consumer "http://localhost:8001/"
                   (fn [event] (println event))
                   {:start-date-time  #inst "2013-05-15T18:00:00"
                    :http-client-opts {:basic-auth ["user" "pass"]}}))

  (println @consumer)

  (stop consumer))
