(ns pulley.consume
  (:require [pulley.state-process  :as state-process]
            [clojure.edn]
            [clj-http.client       :as http]
            [clj-time.core         :as time]
            [clj-time.coerce       :as time-coerce]
            [clj-time.format       :as time-format]
            [cemerick.url          :as url]))

(def ^:private default-http-client-opts
  {:headers {"Accept" "application/edn"}})

(defn- fetch
  [feed-url http-client-opts]
  (let [res (http/get feed-url (merge http-client-opts default-http-client-opts))]
    (if (http/success? res)
      (clojure.edn/read-string (:body res))
      nil)))


;;--------------------------------------------------------------------
;; url helpers
;; TODO: Maybe shared?

(defn- from-url
  [feed-url from]
  (assoc-in feed-url [:query "from"] from))

(defn- from-timestamp-url
  [feed-url timestamp]
  (assoc-in feed-url [:query "from-timestamp"] timestamp))

(defn- from-date-time-url
  [feed-url date-time]
  (assoc-in feed-url [:query "from-date-time"]
            (->> date-time
                 (time-coerce/to-date-time)
                 (time-format/unparse (time-format/formatters :date-time-no-ms)))))


;;--------------------------------------------------------------------
;; consumption
;;

(defmulti consume-state :state)

(defmethod consume-state :wait
  [{:keys [poll-interval] :as state}]
  (when poll-interval
    (Thread/sleep poll-interval))
  (assoc state :state :walk))

(defmethod consume-state :walk
  [{:keys [url handle-event http-client-opts walk-delay] :as state}]
  (try
    (when-let [res (fetch url http-client-opts)]
      (doseq [event (:events res)]
        (handle-event event))
      (if-let [next (:next res)]
        (do
          (Thread/sleep walk-delay)
          (assoc state :state :walk :url next))
        (assoc state :state :wait)))
    (catch Exception e
      (assoc state
        :state :error
        :exception e))))

(defmethod consume-state :error
  [{:keys [exception handle-exception retry-delay] :as state}]
  (when handle-exception
    (try
      (handle-exception exception state)
      (catch Exception e
        ;; TODO: Handle better!
        (println "Exception thrown by handle-exception function")
        (println e))))
  (when retry-delay
    (Thread/sleep retry-delay))
  (-> state
      (assoc :state :walk)
      (dissoc :exception)))


;;
;; initialization
;;

(defn- parse-url
  "Given an URL as a string, return an urly URL. If trailing slash is
  missing and URL has not path, ensure there is a trailing slash."
  [url-string]
  (let [url (url/url url-string)]
    (if (= (:path url) "")
      (assoc url :path "/")
      url)))

(defn- start-url
  [feed-url opts]
  (let [{:keys [start-id start-timestamp start-date-time]} opts
        url (parse-url feed-url)]
    (cond
     start-id        (from-url url start-id)
     start-timestamp (from-timestamp-url url start-timestamp)
     start-date-time (from-date-time-url url start-date-time)
     :default        url)))

(def ^:private default-opts
  {:poll-interval 10000
   :walk-delay    10
   :retry-delay   10000})

(defn- initial-state
  [feed-url handle-event {:keys [handle-exception http-client-opts] :as opts}]
  (-> {:state            :walk
       :url              (str (start-url feed-url opts))
       :handle-event     handle-event
       :handle-exception handle-exception
       :http-client-opts http-client-opts}

      (merge (-> (merge default-opts opts)
                 (select-keys (keys default-opts))))))



(defn make-consumer
  "Creates and starts a process which will consume events from a pulley
  feed, applying a function to each consumed event.

  `feed-url` is the base url of a pulley feed

  `handle-event` is a function that will be applied to each consumed
  event. It should expect a single argument, a map, with the
  keys :id, :timestamp and :data.

  `opts` is a map of options which can be used to control certain
  aspects of the consumption. Currently supported options are:

    :start-id         - start consumption at the event with the given
                        id
    :start-timestamp  - start consumption at the first event that
                        occurred on the given timestamp, in seconds
                        since EPOCH.
    :start-date-time  - start consumption at the first event that
                        occurred on the given date time, as an ISO8601
                        formatted string.
    :poll-interval    - time in milliseconds to wait between polls
    :retry-delay      - time in milliseconds to wait before retrying when
                        in the error state.
    :walk-delay       - time in milliseconds to wait between walking
                        successive pages in the feed. May be used to
                        limit the pressure on the publisher.
    :handle-exception - a function of two args, an Exception and the
                        consumer state, which will be invoked each
                        time the consumer enters the :error state
    :http-client-opts - any extra parameters needed for the clj-http
                        library, for example auth headers. Will be
                        used when requesting data from the feed.
    
  Returns an object that can be stopped, joined or have its current
  state inspected by dereferencing it."
  [feed-url handle-event opts]
  (let [initial-state (initial-state feed-url handle-event opts)]
    (state-process/state-process consume-state initial-state)))

(def stop pulley.state-process/stop)
(def join pulley.state-process/join)

(comment

  (fetch "http://localhost:8001/?from=2")
  (fetch "http://localhost:8001/?from-timestamp=1368726132")
  (fetch "http://localhost:8001/?from-date-time=2013-05-17T11:42")

  (def consumer
    (make-consumer "http://localhost:8001/"
                   println
                   {:start-date-time  #inst "2013-05-15T18:00:00"
                    :polling-interval 10000
                    :retry-delay      1000
                    :handle-exception println
                    :http-client-opts {:basic-auth ["user" "pass"]}}))

  (println @consumer)

  (stop consumer))
