(ns redis.connection
  (:use [redis.protocol :only (write-to-buffer write-to-stream make-inline-command)])
  
  (:import [java.net Socket]
           [java.io BufferedInputStream ByteArrayOutputStream]))

;;; Protocols

(defprotocol RedisConnectionPool
  (get-connection [pool connection-spec])
  (release-connection [pool connection]
                      [pool connection exception]))

(defprotocol RedisConnection
  (get-server-spec [connection])
  (connected? [connection])
  (close [connection])
  (input-stream [connection])
  (output-stream [connection]))


;;; Macros

;; TODO
;; 1. Wait for a Broken pipe
;; 2. Try atom variation and see if we win, in principle.


;;; ORIGINAL
#_(defmacro with-connection [name pool server-spec & body]
  `(let [~name (get-connection ~pool ~server-spec)]
     (try
       ~@body
       (catch Exception e#
         (throw e#))
       (finally
        (release-connection ~pool ~name)))))



;; CURRENT ATTEMPT!!!! **** @@@@
(defmacro with-connection [name pool server-spec & body]
  `(let [~name (get-connection ~pool ~server-spec)]
     (try
       (let [body-result# ~@body]
         (release-connection ~pool ~name)
         body-result#)
       (catch Exception e#
         (release-connection ~pool ~name e#)
         (throw e#)))))



;;; TODO NB New idea, wrapping get-connection with a try

(defmacro with-connection [name pool server-spec & body]
  `(let [;;_# (println name)
         ;;_# (println ~name)
         ~name (try (get-connection ~pool ~server-spec)
                    (catch Exception e#
                      (println (str "CONNECTION EXCEPTION " e#
                                    " " ~pool
                                    ;;" " name
                                    ))
                      ;;~pool
                      ;;~name
                      ;;(str ~pool ~name d#)
                      ;;(release-connection ~pool ~name d#)
                      ))]
     (try
       ~@body
       (catch Exception e#
         (println (str "BODY EXCEPTION " e#))
         (throw e#))
       (finally
        (release-connection ~pool ~name)))))

(defmacro testy [x]
  `(let [
         ~x "hello"
         _# (println ~x)

         ]))

(testy y)



(comment ;; Hackish... follows ObjectPool.html example... seems to work?
  (defmacro with-connection [name pool server-spec & body]
   `(let [~name (get-connection ~pool ~server-spec)
          flag# (atom false)

          ]
      (try
        ~@body
        (catch Exception e#
          (release-connection ~pool ~name e#)
          (reset! flag# true)
          (throw e#))
        (finally
         (when-not @flag#
           (release-connection ~pool ~name)))))))


(comment ;; Loggin
  (defmacro with-connection [name pool server-spec & body]
   `(try

      (let [~name (get-connection ~pool ~server-spec)]
        (try
          ~@body
          (catch Exception e#
            (println (str "body exception " e#))
            (throw e#))
          (finally
           (release-connection ~pool ~name))))
      (catch Exception e#
        (println (str "let exception " e#))))))


(comment
  ;;REF1 http://commons.apache.org/pool/apidocs/org/apache/commons/pool/ObjectPool.html
  ;;REF2 http://groups.google.com/group/clojure/browse_thread/thread/86c87e1fc4b1347c
  (defmacro with-connection [name pool server-spec & body]
    `(try (let [~name (get-connection ~pool ~server-spec)]
            (try
              ~@body
              (catch Exception e#
                (throw e#))
              (finally (release-connection ~pool ~name)))
            (catch Exception e# (release-connection ~pool ~name e#))))))


;;; Implementations
(defn send-command-and-read-reply
  [connection command]
  (let [buf (ByteArrayOutputStream.)
        in (input-stream connection)
        out (output-stream connection)]
    (write-to-buffer command buf)
    (write-to-stream buf out)
    (redis.protocol/read-reply in)))

(defn connection-alive? [connection]
  "Determines whether the connection is still alive"
  (let [ping (make-inline-command "PING")
        resp (send-command-and-read-reply connection ping)]
    (= resp "PONG")))

(defrecord Connection [#^Socket socket server-spec]
  RedisConnection
  (get-server-spec [this] server-spec)
  (connected? [this] (connection-alive? this))
  (close [this] (.close socket))
  (input-stream [this] (BufferedInputStream. (.getInputStream socket)))
  (output-stream [this] (.getOutputStream socket)))

(def default-connection-spec {:host "127.0.0.1"
                              :port 6379
                              :timeout 5000
                              :password nil
                              :db 0})
;TODO: maybe change this on the connection-alive? side instead?
(defn make-connection [server-spec]
  (let [spec (merge default-connection-spec server-spec)
        {:keys [host port timeout password]} spec
        socket (Socket. #^String host #^Integer port)]
    (doto socket
      (.setTcpNoDelay true)
      (.setSoTimeout timeout))
    (let [connection (Connection. socket server-spec)]
      (if (nil? password)
        connection
        (do (send-command-and-read-reply connection (make-inline-command (str "auth " password)))
              connection)))))

(defrecord NonPooledConnectionPool []
  RedisConnectionPool
  (get-connection [this connection-spec]
    (make-connection connection-spec))
  (release-connection [this connection]
    (close connection))
  (release-connection [this connection exception]
    (close connection)))

(defn make-non-pooled-connection-pool []
  (NonPooledConnectionPool.))

