(ns om-tools.dom
  "DOM element constructors for React. Mirrors om.dom namespace"
  (:refer-clojure :exclude [map meta time])
  (:require
   [clojure.string :as str]
   om.dom
         cljs.core)
       
  (:import
   [cljs.tagged_literals JSValue]))

     
(defn clj->js [v]
  (JSValue. v))

(defn camel-case
  "Converts snake-case to camelCase"
  [s]
  (str/replace s #"-(\w)" (comp str/upper-case second)))

(defn opt-key-case
  "Converts attributes that are snake-case and should be camelCase"
  [attr]
  (if (or (< (count attr) 5)
          (#{"data-" "aria-"} (subs attr 0 5)))
    attr
    (camel-case attr)))

(defn opt-key-alias
  "Converts aliased attributes"
  [opt]
  (case opt
    :class :className
    :for :htmlFor
    opt))

(defn format-opt-key [opt-key]
  (-> opt-key
      opt-key-alias
      name
      opt-key-case
      keyword))

(defn format-opt-val [opt-val]
  (if (map? opt-val)
    (clj->js opt-val)
    opt-val))

(defn format-opts
  "Returns JavaScript object for React DOM attributes from opts map"
  [opts]
  (->> opts
       (clojure.core/map
        (fn [[k v]] [(format-opt-key k) (format-opt-val v)]))
       (into {})
       clj->js))

(defn ^boolean literal? [form]
  (not (or (symbol? form)
           (list? form))))

(defn ^boolean possible-coll? [form]
  (or (coll? form)
      (symbol? form)
      (list? form)))

(def form-tags
  '[input textarea option])

(defn el-ctor [tag]
  (if (some (partial = tag) form-tags)
    (symbol "om.dom" (name tag))
    (symbol "js" (str "React.DOM." (name tag)))))

     
(defn object? [x]
  (instance? JSValue x))

(defn element-args [opts children]
  (cond
   (nil? opts) [nil children]
   (map? opts) [(format-opts opts) children]
   (object? opts) [opts children]
   :else [nil (cons opts children)]))

      
                                  
                                                     
                                                 

     
(defn ^:private gen-om-dom-inline-fn [tag]
  `(defmacro ~tag [opts# & children#]
     (let [ctor# '~(el-ctor tag)]
       (if (literal? opts#)
         (let [[opts# children#] (element-args opts# children#)]
           (cond
            (every? (complement possible-coll?) children#)
            `(~ctor# ~opts# ~@children#)

            (and (= (count children#) 1) (vector? (first children#)))
            `(~ctor# ~opts# ~@(-> children# first flatten))

            :else
            `(apply ~ctor# ~opts# (flatten (vector ~@children#)))))
         `(om-tools.dom/element ~ctor# ~opts# (vector ~@children#))))))

(defmacro ^:private gen-om-dom-inline-fns []
  `(do
     ~@(clojure.core/map gen-om-dom-inline-fn (concat om.dom/tags form-tags))))

     
(gen-om-dom-inline-fns)

      
                   
                                                                         
                                         
                                                                               
                       

;;;;;;;;;;;; This file autogenerated from src/om_tools/dom.cljx
