(ns poolgp.distribute
  (:require [clojure.core.async :as async]
            [clojure.java.io :as io])
  (:import [java.net ServerSocket SocketException Socket])
  (:import java.io.PrintWriter)
  (:gen-class))

(def CURRENT-CYCLE (atom 0))

(def OUTGOING-CHAN (async/chan 1000))

(defn- log
  [msg]
  (println "poolgp.distribute =>" msg))

(defn- make-opp-packet
  "make a clojure map for distributing
  opponents to eval workers"
  [indiv id]
  {:indiv indiv ;clojush individual containing :program
   :cycle @CURRENT-CYCLE
   :type :opponent
   :eval-id id}) ;1->total

(defn- make-indiv-packet
  "make a clojure map for distributing
  individual to eval workers"
  [indiv]
  {:indiv indiv ;clojush individual containing :program
   :cycle @CURRENT-CYCLE
   :type :individual
   :eval-id (java.util.UUID/randomUUID)}) ;random

(defn- check-worker-status
  "attempt to reach a worker node (on startup)"
  [hostname port]
  (loop []
    (log (str "Attempting to reach eval host " hostname ":" port))
    (let [status
      (try
        (Socket. hostname port)
        true
        (catch Exception e
          false))]
      (if (not status)
        (do
            (Thread/sleep 5000)
            (recur))))))

(defn- async-distribute-indiv
  "take a socket and an individual and send"
  [indiv host port]
  (future
    (let [client-socket (Socket. host port)]
        (with-open [writer (io/writer client-socket)]
          (.write writer  (str (pr-str indiv) "\n"))
          (.flush writer)))))

(defn- opp-pool-worker
  "start a thread that servers opponent pool requests"
  [socket opponents]
  (future
    (let [outgoing-ops (map make-opp-packet opponents (range))]
      (loop []
        (let [client-socket (.accept socket)
              writer (io/writer client-socket)]
            (try
              (doall (map
                #(.write writer (str (pr-str %) "\n"))
                          outgoing-ops))
              (catch Exception e
                (.printStackTrace e)))
            (.flush writer)
            (.close client-socket))
      (recur)))))

(defn- incoming-socket-worker
  "start a listener for completed individuals"
  [socket]
  (let [client-socket (.accept socket)
        ind (try
              (.readLine (io/reader client-socket))
            (catch SocketException e
              nil))]
    (do
      (.close client-socket)
      ind)))

(defn- unpack-indiv
  "take individuals returned from poolgp workers and strips off extra
  information (returns clojush.individual)"
  [poolgp-indiv]
  ;TODO: aggregate fitness, etc...
  (:indiv poolgp-indiv))

(defn- verify-indiv
  [indiv]
  )

(defn eval-indiv
  "take individual, opponent list,
  server config, evaluate,
  return individual list
  IN: (list clojush.indvidual)
  OUT: (list clojush.individual)
  "
  [indiv opponents config]
  (let
    [incoming-socket (ServerSocket. (:incoming-port config))
     opp-pool-socket (ServerSocket. (:opp-pool-req-p config))]
    (do
      ;wait for connectivity
      (check-worker-status (:host config) (:outgoing-port config))
      (log "Connected to eval worker")
      ;set infinite timeout
      (.setSoTimeout incoming-socket 0)
      (.setSoTimeout opp-pool-socket 0)
      ;worker that responds to opp pool requests
      (opp-pool-worker opp-pool-socket opponents)

      ;server worker that sends individuals out that arrive on outgoing channel
      (async-distribute-indiv indiv (:host config) (:outgoing-port config))

      ;task that listens for incoming individuals
      (unpack-indiv
        (incoming-socket-worker incoming-socket)))))
