(ns polvo.utils.exchanges.time-bomb
  (:require [manifold.deferred :as d]
            [taoensso.timbre :as log]))

(defn time-bomb
  "Creates a time-bomb that will call `callback` after `limit` milliseconds. It can be restarted using `restart!`
  or disarmed with `disarm!`."
  ([limit callback]
   (time-bomb limit callback (gensym "time-bomb")))
  ([limit callback name]
   {:limit    limit
    :callback callback
    :name     name
    :deferred (atom nil)}))

(defn disarm!
  "Disarms a time-bomb."
  [{:keys [deferred] :as tb}]
  (swap! deferred (fn [d]
                    (when d
                      (d/success! d true)
                      nil))))

(defn start!
  "Starts a time-bomb."
  [{:keys [limit callback name deferred] :as tb}]
  (reset! deferred (-> (d/deferred)
                       (d/timeout! limit)
                       (d/on-realized (fn [_]
                                        (log/debug "Bomb" name "successfully disarmed."))
                                      (fn [_]
                                        (log/debug "Bomb" name "exploded. Calling callback.")
                                        (callback)))))
  (log/debug "Bomb" name "started."))

(defn restart!
  "Restarts a time-bomb."
  [tb]
  (doto tb disarm! start!))
