(ns polvo.utils.exchanges.bitcoin-trade
  (:require [clojure.string :as s]
            [clojure.data.json :as json]
            [aleph.http :as http]
            [byte-streams :as bs]
            [camel-snake-kebab
             [core :as csk]
             [extras :as cske]]
            [manifold
             [deferred :as d]
             [stream :as st]]
            [ring.util.codec :refer [form-encode]]))

(def ^:const ^:private base-endpoint "https://api.bitcointrade.com.br/v3/public")

(def ^:const pairs ["BRLBTC" "BRLETH" "BRLLTC" "BRLBCH" "BRLXRP"])

(def ^:const routes
  {
   ; public
   ::ticker          [:public :get "/ticker/"]
   ::orders          [:public :get "/orders/"]
   ::trades          [:public :get "/trades/"]

   ; markets
   ::balance         [:markets :get "/market/balance/"]
   ::summary         [:markets :get "/market/summary/"]
   ::estimated-price [:markets :get "/market/estimated_price/"]
   ::book-orders     [:markets :get "/market/"]
   ::user-orders     [:markets :get "/market/user_orders_list/"]
   ::order           [:markets :post "/market/create_order/"]
   ::cancel-order    [:markets :delete "/market/user_orders/"]

   ; transfers
   ::withdraw-fee    [:transfers :get "/market/balance/"]
   ::deposit-list    [:transfers :get "/market/summary/"]
   ::withdraw-list   [:transfers :get "/market/estimated_price/"]
   ::withdraw        [:transfers :post "/market/"]
   ::sync            [:transfers :post "/market/user_orders_list/"]
   })

(defn- build-url
  [route {:keys [pair] :as params}]
  (let [[access method endpoint] (routes route)]
    (cond
      (= access :public)
      (str base-endpoint "/" pair endpoint "?" (->> (dissoc params :pair)
                                                    (cske/transform-keys csk/->snake_case_string)
                                                    form-encode))

      (and (= access :markets) (= method :get))
      (str base-endpoint endpoint "?" (-> (cske/transform-keys csk/->snake_case_string params)
                                          form-encode))

      :else ""))
  )

(defn- ->key-fn [as-is?] (if as-is? identity csk/->kebab-case-keyword))

(defn http-request
  [route params {:keys [as-is? api-key] :or {as-is? false api-key nil}}]
  (let [[access method] (routes route)
        key-fn      (->key-fn as-is?)
        http-method (case method
                      :get http/get
                      :post http/post
                      :delete http/delete)]
    (d/chain (http-method (build-url route params)
                          {:headers (merge {"Content-Type" "application/json"}
                                           (if (not= access :public)
                                             (if (some? api-key)
                                               {"x-api-key" api-key}
                                               (throw (ex-info "Non-public method requires API key." {})))
                                             {}))})
             :body
             bs/to-string
             #(json/read-str % :key-fn key-fn))))

(defn client
  ([]
   (client {}))
  ([opts]
   (fn [route params]
     (http-request route params opts))))