(ns polvo.utils.exchanges.binance
  "Provides REST and Websocket clients for Binance."
  (:require [clojure.string :as s]
            [clojure.data.json :as json]
            [clojure.math.combinatorics :refer [cartesian-product]]
            [aleph.http :as http]
            [clj-time.core :as t]
            [buddy.core
             [mac :as mac]
             [codecs :as codecs]]
            [byte-streams :as bs]
            [camel-snake-kebab
             [core :as csk]
             [extras :as cske]]
            [manifold.deferred :as d]
            [ring.util.codec :refer [form-encode]]))

; DEPRECATED, need to update and stop using chime

(def ^{:private true :const true}
  rest-base-endpoint "https://api.binance.com")

(def ^{:private true :const true}
  futures-base-endpoint "https://fapi.binance.com")

(def ^{:private true :const true}
  wss-base-endpoint "wss://stream.binance.com:9443")

(def ^{:private true :const true}
  futures-wss-base-endpoint "wss://fstream.binance.com")

; route structure: [endpoint http-method security-type]
(def ^{:const true}
  routes {; REST API, see https://github.com/binance-exchange/binance-official-api-docs/blob/master/rest-api.md
          ::ping                         ["/api/v1/ping" :get :none]
          ::time                         ["/api/v1/time" :get :none]
          ::exchange-info                ["/api/v1/exchangeInfo" :get :none]
          ::depth                        ["/api/v1/depth" :get :none]
          ::trades                       ["/api/v1/trades" :get :none]
          ::historical-trades            ["/api/v1/historicalTrades" :get :market-data]
          ::agg-trades                   ["/api/v1/aggTrades" :get :none]
          ::klines                       ["/api/v1/klines" :get :none]
          ::avg-price                    ["/api/v3/avgPrice" :get :none]
          ::ticker-24hr                  ["/api/v1/ticker/24hr" :get :none]
          ::ticker-price                 ["/api/v3/ticker/price" :get :none]
          ::book-ticker                  ["/api/v3/ticker/bookTicker" :get :none]
          ::create-order                 ["/api/v3/order" :post :trade]
          ::test-order                   ["/api/v3/order/test" :post :trade]
          ::get-order                    ["/api/v3/order" :get :user-data]
          ::cancel-order                 ["/api/v3/order" :delete :trade]
          ::open-orders                  ["/api/v3/openOrders" :get :user-data]
          ::all-orders                   ["/api/v3/allOrders" :get :user-data]
          ::account                      ["/api/v3/account" :get :user-data]
          ::my-trades                    ["/api/v3/myTrades" :get :user-data]
          ::user-stream                  ["/api/v1/userDataStream" :post :user-stream]
          ::heartbeat                    ["/api/v1/userDataStream" :put :user-stream]
          ::close                        ["/api/v1/userDataStream" :delete :user-stream]

          ; Withdraw API, see https://github.com/binance-exchange/binance-official-api-docs/blob/master/wapi-api.md
          ::withdraw                     ["/wapi/v3/withdraw.html" :post :trade]
          ::deposit-history              ["/wapi/v3/depositHistory.html" :get :user-data]
          ::withdraw-history             ["/wapi/v3/withdrawHistory.html" :get :user-data]
          ::deposit-address              ["/wapi/v3/depositAddress.html" :get :user-data]
          ::account-status               ["/wapi/v3/accountStatus.html" :get :user-data]
          ::system-status                ["/wapi/v3/systemStatus.html" :get :system]
          ::api-trading-status           ["/wapi/v3/apiTradingStatus.html" :get :user-data]
          ::dust-log                     ["/wapi/v3/userAssetDribbletLog.html" :get :user-data]
          ::trade-fee                    ["/wapi/v3/tradeFee.html" :get :user-data]
          ::asset-detail                 ["/wapi/v3/assetDetail.html" :get :user-data]
          ::sub-account-list             ["/wapi/v3/sub-account/list.html" :get :user-data]
          ::sub-account-transfer-history ["/wapi/v3/sub-account/transfer/history.html" :get :user-data]
          ::sub-account-transfer         ["/wapi/v3/sub-account/transfer.html" :get :user-data]
          ::sub-account-assets           ["/wapi/v3/sub-account/assets.html" :get :user-data]

          ; Futures API, see https://binance-docs.github.io/apidocs/futures/en/#test-connectivity
          ::ft-ping                      ["/fapi/v1/ping" :get :none]
          ::ft-time                      ["/fapi/v1/time" :get :none]
          ::ft-exchange-info             ["/fapi/v1/exchangeInfo" :get :none]
          ::ft-depth                     ["/fapi/v1/depth" :get :none]
          ::ft-trades                    ["/fapi/v1/trades" :get :none]
          ::ft-historical-trades         ["/fapi/v1/historicalTrades" :get :market-data]
          ::ft-agg-trades                ["/fapi/v1/aggTrades" :get :none]
          ::ft-klines                    ["/fapi/v1/klines" :get :market-data]
          ::mark-price                   ["/fapi/v1/premiumIndex" :get :none]
          ::funding-rate                 ["/fapi/v1/fundingRate" :get :market-data]
          ::ft-ticker-24h                ["/fapi/v1/ticker/24h" :get :none]
          ::ft-ticker-price              ["/fapi/v1/ticker/price" :get :none]
          ::ft-book-ticker               ["/fapi/v1/ticker/bookTicker" :get :none]
          ::liquidation-orders           ["/fapi/v1/allForceOrders" :get :none]
          ::open-interest                ["/fapi/v1/openInterest" :get :none]
          ::leverage-bracket             ["/fapi/v1/leverageBracket" :get :market-data]})

(defn- encode-sign-params
  "Encodes parameters and adds a HMAC SHA256 signature"
  [api-secret params]
  (let [encoded   (-> {:timestamp (System/currentTimeMillis)}
                      (merge params)
                      form-encode)
        signature (-> encoded
                      (mac/hash {:key api-secret
                                 :alg :hmac+sha256})
                      codecs/bytes->hex)]
    (str encoded "&signature=" signature)))

(defn- fapi?
  "Tells whether a route is from fapi."
  [route]
  (= \f (-> (routes route) first (get 1))))

(defn- base-endpoint
  "Gets correct base endpoint for a given route."
  [route]
  (if (fapi? route)
    futures-base-endpoint
    rest-base-endpoint))

(defn request
  "Sends a request to Binance REST APIs. Kebab case param keys are transformed to camel case automatically.
  Response body keys are transformed to kebab case by default."
  [route {:keys [params key secret as-is?]
          :or   {params {} key nil secret nil as-is? false}}]
  (let [[endpoint method security] (routes route)
        http-method  (case method
                       :post http/post
                       :get http/get
                       :put http/put
                       :delete http/delete)

        camel-params (cske/transform-keys csk/->camelCaseString params)
        query-str    (if (contains? #{:trade :user-data} security)
                       (encode-sign-params secret camel-params)
                       (form-encode camel-params))
        key-fn       (if as-is? identity csk/->kebab-case-keyword)

        url          (str (base-endpoint route) endpoint "?" query-str)
        req          (if key {:headers {"X-MBX-APIKEY" key}} {})]

    (d/chain (http-method url req)
             :body
             bs/to-string
             #(json/read-str % :key-fn key-fn))))

(defn rest-client
  "A closure over fn request allowing to provide default args for api-key, api-secret and as-is?
  Returning function may be called with a param map or named args."
  [options]
  (fn [route params]
    (request route (merge {:params params} options))))

(defn- is-interval? [x]
  (some #{:1m :3m :5m :15m :30m :1h :2h :4h :6h :8h :12h :1d :3d :1w :1M} [(keyword x)]))

(defn- as-symbol [x] (-> x name s/lower-case))

(defn- vec->stream-name
  ([_] "!miniTicker@arr")
  ([t symb] (str (as-symbol symb) "@" (csk/->camelCaseString t)))
  ([_ symb arg2]
   (if-let [interval (is-interval? arg2)]
     (str (as-symbol symb) "@kline_" (name interval))
     (str (as-symbol symb) "@depth" (Integer. arg2)))))

(defn- expand-vec [t & args]
  (->> args
       (map #(if (coll? %) % [%]))
       (apply cartesian-product)
       (map #(apply vec->stream-name t %))))

(defn expand [x]
  (if (string? x) [x] (apply expand-vec x)))

(defn streams [names]
  (let [url (->> names
                 (mapcat expand)
                 (s/join "/")
                 (str wss-base-endpoint "/stream?streams="))]
    (http/websocket-client url)))

(defn stream [name]
  (let [url (str wss-base-endpoint "/ws/" name)]
    (http/websocket-client url)))

;(defn user-stream [cli opts]
;  (let [listen-key (:listen-key (cli :user-stream))
;        s          (stream listen-key opts)
;        ss         (->> (range 1 48)
;                        (map #(-> % (* 30) t/minutes t/from-now)))
;        sched      (chime-at ss (fn [_]
;                                  (cli :heartbeat :listen-key listen-key)))]
;    {:stream     s
;     :listen-key listen-key}))