(ns polvo.utils.exchanges.bitmex
  "Provides REST and Websocket clients for Bitmex."
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [clojure.data.json :as json]
            [clojure.math.numeric-tower :as math]
            [aleph.http :as http]
            [buddy.core
             [mac :as mac]
             [codecs :as codecs]]
            [byte-streams :as bs]
            [camel-snake-kebab
             [core :as csk]
             [extras :as cske]]
            [manifold
             [deferred :as d]
             [stream :as st]]
            [ring.util.codec :refer [form-encode]]))

(def ^:private ^:const
  rest-base-url {:testnet "https://testnet.bitmex.com"
                 :prod    "https://www.bitmex.com"})

(def ^:private ^:const
  ws-base-url {:prod    "wss://www.bitmex.com/realtime"
               :testnet "wss://testnet.bitmex.com/realtime"})

(def ^:private ^:const
  multiplex-base-url {:prod    "wss://www.bitmex.com/realtimemd"
                      :testnet "wss://testnet.bitmex.com/realtimemd"})

(def write-str #(json/write-str % :escape-slash false))

(def ^:private ^:const
  api-endpoint "/api/v1")

(def ^:private
  kw->method {:get    http/get
              :post   http/post
              :put    http/put
              :delete http/delete})

(defn- method-name [kw] (str/upper-case (name kw)))

(def ^:const
  routes
  "Relationship between keywords and API endpoints. See https://www.bitmex.com/api/explorer/"
  {
   ; Announcement
   ::announcement        [:get "/announcement"]
   ::urgent              [:get "/announcement/urgent"]

   ; APIKey
   ::api-key             [:get "/apiKey"]

   ; Chat
   ::get-chat            [:get "/chat"]
   ::post-chat           [:post "/chat"]
   ::chat-channels       [:get "/chat/channels"]
   ::chat-connected      [:get "/chat/connected"]

   ; Execution
   ::execution           [:get "/execution"]
   ::trade-history       [:get "/execution/tradeHistory"]

   ; Funding
   ::funding             [:get "/funding"]

   ; Notification
   ::notification        [:get "/globalNotification"]

   ; Instrument
   ::instrument          [:get "/instrument"]
   ::active-instrument   [:get "/instrument/active"]
   ::active-and-indices  [:get "/instrument/activeAndIndices"]
   ::active-intervals    [:get "/instrument/activeIntervals"]
   ::composite-index     [:get "/instrument/compositeIndex"]
   ::indices             [:get "/instrument/indices"]

   ; Insurance
   ::insurance           [:get "/insurance"]

   ; Leaderboard
   ::leaderboard         [:get "/leaderboard"]
   ::leaderboard-name    [:get "/leaderboard/name"]

   ; Liquidation
   ::liquidation         [:get "/liquidation"]

   ; Order
   ::get-order           [:get "/order"]
   ::edit-order          [:put "/order"]
   ::place-order         [:post "/order"]
   ::delete-order        [:delete "/order"]
   ::delete-all-orders   [:delete "/order/all"]
   ::edit-orders         [:put "/order/bulk"]
   ::place-orders        [:post "/order/bulk"]
   ::cancel-all-after    [:post "/order/cancelAllAfter"]
   ::close-position      [:post "/order/closePosition"]

   ; OrderBook
   ::order-book-l2       [:get "/orderBook/L2"]

   ; Position
   ::position            [:get "/position"]
   ::isolate-position    [:post "/position/isolate"]
   ::leverage-position   [:post "/position/leverage"]
   ::risk-limit          [:post "/position/riskLimit"]
   ::transfer-margin     [:post "/position/transferMargin"]

   ; Quote
   ::quote               [:get "/quote"]
   ::bucketed-quotes     [:get "/quote/bucketed"]

   ; Schema
   ::schema              [:get "/schema"]
   ::websocket-help      [:get "/schema/websocketHelp"]

   ; Settlement
   ::settlement          [:get "/settlement"]

   ; Stats
   ::stats               [:get "/stats"]
   ::stats-history       [:get "/stats/history"]
   ::stats-history-usd   [:get "/stats/history/usd"]

   ; Trade
   ::trade               [:get "/trade"]
   ::bucketed-trades     [:get "/trade/bucketed"]

   ; User
   ::user                [:get "/user"]
   ::affiliate-status    [:get "/user/affiliateStatus"]
   ::cancel-withdrawal   [:post "/user/cancelWithdrawal"]
   ::referral-code       [:get "/user/checkReferralCode"]
   ::commission          [:get "/user/commission"]
   ::communication-token [:post "/user/communicationToken"]
   ::confirm-email       [:post "/user/confirmEmail"]
   ::confirm-withdrawal  [:post "/user/confirmWithdrawal"]
   ::deposit-address     [:get "/user/depositAddress"]
   ::execution-history   [:get "/user/executionHistory"]
   ::logout              [:post "/user/logout"]
   ::margin              [:get "/user/margin"]
   ::min-withdrawal-fee  [:get "/user/minWithdrawalFee"]
   ::preferences         [:post "/user/preferences"]
   ::quote-fill-ratio    [:get "/user/quoteFillRatio"]
   ::request-withdrawal  [:post "/user/requestWithdrawal"]
   ::wallet              [:get "/user/wallet"]
   ::wallet-history      [:get "/user/walletHistory"]
   ::wallet-summary      [:get "/user/walletSummary"]

   ; UserEvent
   ::user-event          [:get "/userEvent"]})

(defn- signature [api-secret verb path expires data]
  (let [data-str (if (empty? data)
                   ""
                   (write-str data))
        full-str (str verb path expires data-str)]

    (-> full-str
        (mac/hash {:key api-secret
                   :alg :hmac+sha256})
        codecs/bytes->hex)))

(defn- http-request [^String api-key ^String api-secret expiration as-is? testnet? route query data]
  (let [expires   (int (math/ceil (/ (+ (System/currentTimeMillis) expiration) 1000)))
        [verb-kw path] (routes route)

        fix-query (fn [query]
                    (->> query
                         (map (fn [[key val]]
                                [key (as-> val val'
                                           (if (contains? #{:columns "columns"} key)
                                             (->> val' (mapv #(if (keyword? %)
                                                                (csk/->camelCaseString %)
                                                                %)))
                                             val')

                                           (if (contains? #{:filter "filter"} key)
                                             (cske/transform-keys csk/->camelCaseString val')
                                             val')

                                           (if (and (seqable? val') (not (string? val')))
                                             (write-str val')
                                             val'))]))
                         (into {})))

        full-path (str api-endpoint path (if (empty? query)
                                           ""
                                           (str "?" (form-encode (fix-query query)))))
        sign      (signature api-secret (method-name verb-kw) full-path expires data)
        method    (kw->method verb-kw)
        key-fn    (if as-is? identity csk/->kebab-case-keyword)
        base-url  (if testnet? (rest-base-url :testnet) (rest-base-url :prod))]

    (d/chain (method (str base-url full-path) {:headers       {"api-expires"   expires
                                                               "api-key"       api-key
                                                               "api-signature" sign}
                                               :save-request? true})
             :body
             bs/to-string
             #(json/read-str % :key-fn key-fn))))

(defn rest-client
  "Creates an http client for bitmex with the given keys

  Options:
  `expiration`: expiration in milliseconds. See https://www.bitmex.com/app/apiKeysUsage
  `as-is?`: when `false` does not convert keys to kebab-case keywords
  `testing?`: if `true`, uses testnet"
  [{:keys [key secret expiration as-is? testnet?]
    :or   {expiration 5000
           as-is?     false
           testnet?   false}
    :as   opts}]
  (partial http-request key secret expiration as-is? testnet?))

(s/def ::subscription (s/or :string string?
                            :keyword keyword?
                            :pair (s/cat :sub-kw keyword? :symbol (s/or :keyword keyword? :string string?))))

(defn- kw->sub [kw]
  (if (= kw :order-book-l2-25)
    "orderBookL2_25"
    (csk/->camelCaseString kw)))

(defn expand
  "Converts keywords and keyword-instrument pairs to subscription strings.
  Subscription follows `::subscription` spec."
  [subscription]
  (let [[kind sub] (s/conform ::subscription subscription)]
    (case kind
      :string sub
      :keyword (kw->sub sub)
      :pair (let [{:keys [sub-kw symbol]} sub
                  [kind instrument] symbol
                  instrument-str (case kind
                                   :keyword (str/upper-case (name instrument))
                                   :string (str/upper-case instrument))]
              (str (kw->sub sub-kw) ":" instrument-str)))))

(defn- subscribe*
  "Generates request map for subscription"
  [subscriptions]
  {:op   "subscribe"
   :args (mapv expand subscriptions)})

(defn subscribe!
  "Subscribes. Subscriptions are expanded with `expand`."
  [s subscriptions]
  (st/put! s (write-str (subscribe* subscriptions))))

(defn- unsubscribe*
  "Unsubscribes. Subscriptions are expanded with `expand`."
  [subscriptions]
  {:op   "unsubscribe"
   :args (mapv expand subscriptions)})

(defn unsubscribe!
  "Unsubscribes. Subscriptions are expanded with `expand`."
  [s subscriptions]
  (st/put! s (write-str (unsubscribe* subscriptions))))

(defn- authenticate*
  "Authenticates with given key an secret."
  [api-key api-secret]
  (let [expiration 5000
        expires-at (int (math/ceil (/ (+ (System/currentTimeMillis) expiration) 1000)))]

    {:op   "authKeyExpires"
     :args [api-key expires-at
            (signature api-secret "GET" "/realtime" expires-at {})]}))

(defn authenticate!
  "Authenticates with given key an secret."
  [s api-key api-secret]
  (st/put! s (write-str (authenticate* api-key api-secret))))

(defn ws-client
  "Returns an deferred yielding a duplex `manifold` stream connected to BitMEX websocket API.

  Options:
  `as-is?`: when `false` does not convert keys to kebab-case keywords.
  `testnet?`: if `true`, uses testnet.
  `subscriptions`: a sequence of subscriptions. They are expanded using `expand`."

  [{:keys [key secret as-is? testnet? subscriptions]
    :or   {as-is?        false
           testnet?      false
           subscriptions []}
    :as   opts}]

  (let [full-url (if (empty? subscriptions)
                   (if testnet? (ws-base-url :testnet) (ws-base-url :prod))
                   (str ws-base-url "?subscribe=" (str/join "," (mapv expand subscriptions))))]
    (d/let-flow [s       (http/websocket-client full-url {:max-frame-payload 1000000
                                                          :send-after-idle   5000})
                 spliced (st/splice s (st/map (fn [msg]
                                                (if as-is?
                                                  (json/read-str msg)
                                                  (json/read-str msg :key-fn csk/->kebab-case-keyword)))
                                              s))
                 _ (when (and (some? key) (some? secret))
                     (authenticate! s key secret))]
      spliced)))

(s/def ::key string?)
(s/def ::secret string?)
(s/def ::id string?)
(s/def ::topic string?)
(s/def ::subscriptions (s/coll-of ::subscription))

(s/def ::account (s/keys :req-un [::id ::topic]
                         :opt-un [::key ::secret ::subscriptions]))

(defn mp-open-stream!
  "Opens a stream in a multiplex connection."
  [s id topic]
  (st/put! s (write-str [1 id topic])))

(defn mp-close-stream!
  "Closes a stream in a multiplex connection."
  [s id topic]
  (st/put! s (write-str [2 id topic])))

(defn mp-authenticate!
  "Authenticates an account in a multiplex connection."
  [s id topic key secret]
  (st/put! s (write-str [0 id topic (authenticate* key secret)])))

(defn mp-subscribe!
  "Subscribes to topics in a multiplex connection."
  [s id topic subscriptions]
  (st/put! s (write-str [0 id topic (subscribe* subscriptions)])))

(defn mp-unsubscribe!
  "Unsubscribes to topics in a multiplex connection."
  [s id topic subscriptions]
  (st/put! s (write-str [0 id topic (unsubscribe* subscriptions)])))

(defn mp-client
  "Creates and initializes a multiplex connection."
  [accounts {:keys [as-is? testnet?]
             :or   {as-is? false testnet? false}}]
  {:pre [(s/valid? (s/coll-of ::account) accounts)]}

  (let [full-url (if testnet?
                   (multiplex-base-url :testnet)
                   (multiplex-base-url :prod))]
    (d/let-flow [s      (http/websocket-client full-url {:max-frame-payload 1000000
                                                         :send-after-idle   5000})
                 inst-s (st/splice s (st/map (fn [msg]
                                               (if as-is?
                                                 (json/read-str msg)
                                                 (json/read-str msg :key-fn csk/->kebab-case-keyword)))
                                             s))]

      (doseq [{::keys [key secret id topic subscriptions]} accounts]
        (d/chain (mp-open-stream! inst-s id topic)

                 (fn [_]
                   (when (and (some? key) (some? secret))
                     (mp-authenticate! inst-s id topic key secret)))

                 (fn [_]
                   (when-not (empty? subscriptions)
                     (mp-subscribe! inst-s id topic subscriptions)))))

      inst-s)))