(ns piotr-yuxuan.dove.dove
  (:require [byte-streams]
            [camel-snake-kebab.core :as csk])
  (:import (org.apache.avro Schema$Type Schema Schema$FixedSchema Schema$Field Schema$UnionSchema Schema$MapSchema Schema$RecordSchema Schema$ArraySchema)
           (org.apache.avro.util Utf8)
           (java.nio ByteBuffer)))

(defmulti decode
  (fn [_ ^Schema avro-schema]
    (.getType avro-schema)))

(defn schema-properties
  [config ^Schema avro-schema]
  (let [schema-type (csk/->kebab-case-keyword (str (.getType avro-schema)))]
    (merge {:avro/type schema-type}
           (when-let [schema-name (try (.getName avro-schema) (catch Throwable _))] {:avro/name schema-name})
           (when-let [schema-namespace (try (.getNamespace avro-schema) (catch Throwable _))] {:avro/namespace schema-namespace})
           (when-let [schema-fullname (try (.getFullName avro-schema) (catch Throwable _))] {:avro/fullname schema-fullname})
           (when-let [doc (.getDoc avro-schema)] {:description doc})
           (when-let [logical-type (.getLogicalType avro-schema)] {:logical-type logical-type})
           (.getObjectProps avro-schema))))

(defmethod decode Schema$Type/RECORD
  [config ^Schema$RecordSchema avro-schema]
  (println ::debug "Schema$Type/RECORD")
  (->> (.getFields avro-schema)
       (map (fn [^Schema$Field schema-field]
              (let [properties (merge {:position (.pos schema-field)
                                       :order (csk/->kebab-case-keyword (str (.order schema-field)))}
                                      (let [aliases (try (.aliases schema-field) (catch Throwable _))]
                                        (when-not (empty? aliases)
                                          {:aliases (.aliases schema-field)}))
                                      (when (.hasDefaultValue schema-field)
                                        {:default (.defaultVal schema-field)}))]
                [(.name schema-field)
                 properties
                 (decode config (.schema schema-field))])))
       (into [:map (merge {}
                          (schema-properties config avro-schema))])))

(defmethod decode Schema$Type/ENUM
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/ENUM")
  [:fn (schema-properties config avro-schema)
   string?])

(defmethod decode Schema$Type/ARRAY
  [config ^Schema$ArraySchema avro-schema]
  (println ::debug "Schema$Type/ARRAY")
  [:sequential (schema-properties config avro-schema) (decode config (.getElementType avro-schema))])

(defmethod decode Schema$Type/MAP
  [config ^Schema$MapSchema avro-schema]
  (println ::debug "Schema$Type/MAP")
  [:map-of (schema-properties config avro-schema)
   string?
   (decode config (.getValueType avro-schema))])

(defmethod decode Schema$Type/UNION
  [config ^Schema$UnionSchema avro-schema]
  (println ::debug "Schema$Type/UNION")
  (->> (.getTypes avro-schema)
       (map (partial decode config))
       (into [:or (schema-properties config avro-schema)])))

(defmethod decode Schema$Type/FIXED
  [config ^Schema$FixedSchema avro-schema]
  (println ::debug "Schema$Type/FIXED")
  [:and (schema-properties config avro-schema)
   [:fn #(instance? ByteBuffer %)]
   [:fn (fn [^ByteBuffer byte-buffer]
          (= (.getFixedSize avro-schema)
             (.capacity byte-buffer)))]])

(defn utf8? [data] (instance? Utf8 data))

(defmethod decode Schema$Type/STRING
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/STRING")
  [:or (schema-properties config avro-schema)
   string?
   [:fn utf8?]])

(defmethod decode Schema$Type/BYTES
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/BYTES")
  [bytes? (schema-properties config avro-schema)])

(defmethod decode Schema$Type/INT
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/INT")
  [int? (schema-properties config avro-schema)])

(defmethod decode Schema$Type/LONG
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/LONG")
  [:fn (schema-properties config avro-schema) #(instance? Long %)])

(defmethod decode Schema$Type/FLOAT
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/FLOAT")
  [float? (schema-properties config avro-schema)])

(defmethod decode Schema$Type/DOUBLE
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/DOUBLE")
  [double? (schema-properties config avro-schema)])

(defmethod decode Schema$Type/BOOLEAN
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/BOOLEAN")
  [boolean? (schema-properties config avro-schema)])

(defmethod decode Schema$Type/NULL
  [config ^Schema avro-schema]
  (println ::debug "Schema$Type/NULL")
  [nil? (schema-properties config avro-schema)])

;; [malli.json-schema :as json-schema]

(defn ^Schema encode
  [malli-schema])
