(ns debux
  (:require (clojure [string :as str]
                     [pprint :as pp] )))

(defmacro ^:private dbg_
  "The internal macro to debug dbg macro.
   <form any> a form to be evaluated"
  [form]
  `(let [return# ~form]
     (println ">>> dbg_:" '~form "=>" return# "<<<")
     return#))

(defn- vec->map
  "Transforms a vector into an array-map with key/value pairs.
   <v [<form any>*]>

  (def a 10)
  (def b 20)
  (vec-map [a b :c [30 40]])
  => {:a 10 :b 20 ::c :c :[30 40] [30 40]}"
  [v]
  (apply array-map
         (mapcat (fn [elm]
                   `[~(keyword (str elm)) ~elm])
                 v) ))

(defn- parse-args
  "Parses <args> into a map.
   <args (<arg any>*)>
   <return {}>"
  [args]
  (loop [args args
         acc {}]
    (let [f (first args)
          s (second args)]
      (cond
        (empty? args)
        acc

        (number? f)
        (recur (next args) (assoc acc :n f))

        (string? f)
        (recur (next args) (assoc acc :msg f))

        (= f :if)
        (recur (nnext args) (assoc acc :condition s)) ))))

(defmacro dbg
  "The macro for debuggng and analyzing Clojure source code.
   <form any> a form to be evaluated
   <args (<arg any>)> the arguments to control the evaluation way"
  [form & args]
  (let [form# (if (vector? form) (vec->map form) form)
        args# (parse-args args)
        n#    (:n args#)
        msg#  (:msg args#)]
    `(binding [*print-length* (or ~n# 100)]
       (let [orig-form# '~form
             format#    (str ">> dbg: " orig-form#
                             (and ~msg# (str "   <" ~msg# ">"))
                             " =>%n%s <<%n")
             return#    ~form#
             return#    (if ~n# (take ~n# return#) return#)
             pprint#    (str/trim (with-out-str (pp/pprint return#)))
             condition# ~(:condition args#)]
         (when (or (nil? condition#) condition#)
           (printf format# pprint#)
           (flush))
         (if (vector? orig-form#)
           ~form
           return#) ))))
