(ns cljs.debux
  (:require [clojure.string :as str]))

;; These vars are actually defined in cljs/debux.cljs file. 
(declare changed? print-log get-style style*)

(defmacro ^:private dbg_
  "The internal macro to debug dbg macro.
   <form any> a form to be evaluated"
  [form]
  `(let [return# ~form]
     (.log js/console (str ">>> dbg_: " '~form "=> " return# " <<<"))
     return#))

(defn- vec->map
  "Transforms a vector into an array-map with key/value pairs.
   <v [<form any>*]>
   <rerurn {}>

  (def a 10)
  (def b 20)
  (vec-map [a b :c [30 40]])
  => {:a 10 :b 20 ::c :c :[30 40] [30 40]}"
  [v]
  (apply array-map
         (mapcat (fn [elm]
                   `[~(keyword (str elm)) ~elm])
                 v) ))

(defn- parse-args
  "Parses <args> into a map.
   <args (<arg any>*)>
   <return {}>"
  [args]
  (loop [args args
         acc {}]
    (let [f (first args)
          s (second args)]
      (cond
        (empty? args)
        acc

        (number? f)
        (recur (next args) (assoc acc :n f))

        (string? f)
        (recur (next args) (assoc acc :msg f))

        (= f :if)
        (recur (nnext args) (assoc acc :condition s))

        (= f :js)
        (recur (next args) (assoc acc :js true))

        (#{:once :o} f)
        (recur (next args) (assoc acc :once true))

        (#{:style :s} f)
        (recur (nnext args) (assoc acc :style s)) ))))

(defmacro dbg
  "The macro for debuggng and analyzing ClojureScript source code
   in a browser REPL like weasel.
   <form any> a form to be evaluated
   <args (<arg any>*)> the arguments to control the evaluation way"
  [form & args]
  (let [form# (if (vector? form) (vec->map form) form)
        args# (parse-args args)
        n#    (:n args#)
        msg#  (:msg args#)
        once# (:once args#)]
    `(let [orig-form# '~form
           header#    (str ">> dbg: " orig-form#
                           (and ~msg# (str "   <" ~msg# ">"))
                           " =>\n")
           return#    ~form#
           return#    (if ~n# (take ~n# return#) return#)
           pprint#    (str/trim (with-out-str (cljs.pprint/pprint return#)))
           condition# ~(:condition args#)]
       (when (or (nil? condition#) condition#)
         (cond
           ~once#
           (when (changed? (str orig-form# " " '~args#) (str return#))
             (print (str header# pprint# " << (:once mode)\n")))
            
           :else
           (print (str header# pprint# " <<\n" )) ))
       (flush)
       (if (vector? orig-form#)
         ~form
         return#) )))

(defmacro clog
  "The macro for debuggng and analyzing ClojureScript source code
   in a browser's console window.
   <form any> a form to be evaluated
   <args (<arg any>*)> the arguments to control the evaluation way"
  [form & args]
  (let [form# (if (vector? form) (vec->map form) form)
        args# (parse-args args)
        n#    (:n args#)
        msg#  (:msg args#)
        once# (:once args#)
        js#   (:js args#)
        style# (:style args#)]
    `(let [orig-form# '~form
           header#     (str "%cclog: %c " orig-form#
                            " %c" (and ~msg# (str "   <" ~msg# ">"))
                            " =>\n")
           form-style# (or (get-style ~style#) (get-style :debug))
          
           return#     ~form#
           return#     (if ~n# (take ~n# return#) return#)
           pprint#     (str/trim (with-out-str (cljs.pprint/pprint return#)))
           body#       (str pprint# (and ~once# "   (:once mode)")
                            (and ~js# "\n%O"))
           condition#  ~(:condition args#)]
       (when (or (nil? condition#) condition#)
         (cond
           ~once#
           (when (changed? (str orig-form# " " '~args#) (str return#))
             (print-log header# form-style# body# ~js# return#))

           :else
           (print-log header# form-style# body# ~js# return#) ))

       (if (vector? orig-form#)
         ~form
         return#) )))


(defmacro break
  "Sets a break point."
  ([] '(js* "debugger;"))
  ([kw condition]
     `(if (and (= ~kw :if) ~condition)
        ~'(js* "debugger;") )))
