# debux

Debux is a simple library for debugging Clojure(Script) source code. I wrote this library to debug my own Clojure(Script) souce code and to analyze other developer's Clojure(Script) souce code. 


## Prerequisites

* clojure 1.7.0 or later
* clojurescript 0.0-3308 or later


## Installation

To include *debux* in your project, simply add the following to your *project.clj* dependencies:


```
[philoskim/debux "0.1.0"]
```


## Usage in Clojure

In the first place, the following line has to be included in your file to use *debux* library.


```clojure
(use 'debux)
```


### Basic usage

This is a simple example. The macro `dbg` prints an original form and pretty-prints the evaluated value on the REPL window. Then it returns the value without interrupting code execution.


```clojure
(* 2 (dbg (+ 10 20)))
; => 60
```

```
>> dbg: (+ 10 20) =>
30 <<
```

The `dbg` macro can be nested.

```clojure
(dbg (* 2 (dbg (+ 10 20))))
; => 60
```

```
>> dbg: (+ 10 20) =>
30 <<
>> dbg: (* 2 (dbg (+ 10 20))) =>
60 <<
```

Sometimes you need to see several forms evaluated. To do so, a literal vector form can be used like this.

```clojure
(let [a (take 5 (range))
      {:keys [b c d] :or {d 10 b 20 c 30}} {:c 50 :d 100}
      [e f g & h] ["a" "b" "c" "d" "e"]]
  (dbg [a b c d e f g h]))
; => [(0 1 2 3 4) 20 50 100 "a" "b" "c" ("d" "e")]
```

```
>> dbg: [a b c d e f g h] =>
{:a (0 1 2 3 4),
 :b 20,
 :c 50,
 :d 100,
 :e "a",
 :f "b",
 :g "c",
 :h ("d" "e")} <<
```

Notice that the printed value is a map, not a vector and the form is prepended with colon to differenciate the form from the evaluated value.

Further examples:

```clojure
(def a 10)
(def b 20)

(dbg [a b :c [30 40]])
; => [10 20 :c [30 40]]
```

```
>> dbg: [a b :c [30 40]] =>
{:a 10, :b 20, ::c :c, :[30 40] [30 40]} <<
```


### Various options

The various options can be added and combinated in any order after the form.


#### String option

You can add your own message in a string and it will be printed betwen less-than and more-than sign like this.


```clojure
(dbg (repeat 5 (dbg (repeat 5 "x")
                    "inner repeat"))
     "outer repeat")
; => (("x" "x" "x" "x" "x")
;     ("x" "x" "x" "x" "x")
;     ("x" "x" "x" "x" "x")
;     ("x" "x" "x" "x" "x")
;     ("x" "x" "x" "x" "x"))
```

```
>> dbg: (repeat 5 "x")   <inner repeat> =>
("x" "x" "x" "x" "x") <<
>> dbg: (repeat 5 (dbg (repeat 5 "x") "inner repeat"))   <outer repeat> =>
(("x" "x" "x" "x" "x")
 ("x" "x" "x" "x" "x")
 ("x" "x" "x" "x" "x")
 ("x" "x" "x" "x" "x")
 ("x" "x" "x" "x" "x")) <<
```


#### Number option

You can add a number to limit the number of the elemets in evaluating an infinite lazy-seq and will avoid the stack-overflow exception.


```clojure
(dbg (range) 5)
; => (0 1 2 3 4)
```

```
>> dbg: (range) =>
(0 1 2 3 4) <<
```

If you omit the number in evaluating an infinite lazy-seq, it will print default 100 elements to avoid the stack-overflow exception.

```clojure
(take 5 (dbg (range)))
; => (0 1 2 3 4)
```

```
>> dbg: (range) =>
(0 1 2 3 4 5 6 7 8 9
 ......
 98 99 ...) <<
```

#### `:if` `expression` option

You can set `:if` `expression` like this.

```clojure
(for [i (range 10)]
  (dbg i :if (even? i)))
; => (0 1 2 3 4 5 6 7 8 9)
```

```
>> dbg: i =>
0 <<
>> dbg: i =>
2 <<
>> dbg: i =>
4 <<
>> dbg: i =>
6 <<
>> dbg: i =>
8 <<
```

## Usage in ClojureScript on Browser Console

In ClojureScript, you should use `clog` instead of `dbg`, because `clog` macro uses the `console.log` fuction of browser's developer tools to style the form.


The following `(:require ...)` line has to be included in your file to use *debux* library in ClojureScript.


```clojure
(ns example.core
  (:require [cljs.debux :as d :refer-macros [clog dbg break]]))


(clog (repeat 5 (clog (repeat 5 "x")
                      "inner repeat"))
      "outer repeat")
```

![clog-1](doc/img/clog-1.png)


Besides 'Usage in Clojure' features, you can use additional ones in `clog` macro.

### CSS Styling

#### Predefined style keywords

You can style the form, using the following predefined keywords.

| keyword | abbreviation |
| ------- |:------------:|
| :style  | :s           |
| :error  | :e           |
| :warn   | :w           |
| :info   | :i           |
| :debug  | :d           |


```
(clog (+ 10 20) :style :error "error style")
(clog (+ 10 20) :style :warn "warn style")
(clog (+ 10 20) :style :info "info style")
(clog (+ 10 20) :style :debug "debug style")
(clog (+ 10 20) "debug style is default")
```

![clog-2](doc/img/clog-2.png)


#### User-defined style

You can redefine the predefined styles or define your own new style like this.


```clojure
(d/merge-style {:warn "background: #9400D3; color: white"
                :love "background: #FF1493; color: white"})

(clog (+ 10 20) :style :warn "warn style changed")
(clog (+ 10 20) :style :love "love style")

;; You can style the form directly in string format in any way you want.
(clog (+ 10 20) :style "color:orange; background:blue; font-size: 14pt")
```

![clog-3](doc/img/clog-3.png)


### `:once` option

If you add `:once` (or `:o` in brief) option after the form, the same evaluated value will not be printed. This is a very useful feature, when you are debugging a game programming, where successive multiple frames usually have the same evaluated value.


```clojure
(clog (+ 30 40) :once)

;; The following will not be printed, because it has :once option and
;; (+ 30 40) will have the same evaluated value as the preceeding one.
(clog (+ 30 40) :once)

```

![clog-4](doc/img/clog-4.png)


### `:js` option

If `:js` option is added after the form, the JavaScript object will be printed as well, so you can inspect the internal structures of ClojureScript data types.

```
(clog {:a 10 :b 20} :js)
```

![clog-5](doc/img/clog-5.png)


## Usage in ClojureScript on Browser REPL

You can use `dbg` macro on the browser REPL like *weasel*. The following is an example about runing browser REPL *weasel* and using `dbg` macro.


```clojure
;; project.clj
(defproject example "0.1.0-SNAPSHOT"
  :dependencies [[org.clojure/clojure "1.7.0"]
                 [org.clojure/clojurescript "0.0-3308"]
                 [com.cemerick/piggieback "0.2.1"]
                 [weasel "0.7.0"]
                 [philoskim/debux "0.1.0"]]
  :plugins [[lein-cljsbuild "1.0.5"]
            [lein-figwheel "0.3.7"]]
  :source-paths ["src/clj"]
  :clean-targets ^{:protect false}
                 ["resources/public/js/app.js"
                  "resources/public/js/app.js.map"]
  :repl-options {:nrepl-middleware [cemerick.piggieback/wrap-cljs-repl]}
  :cljsbuild {:builds [{:id "dev"
                        :source-paths ["src/cljs"]
                        :figwheel true
                        :compiler {:main "example.brepl"
                                   :asset-path "js/out"
                                   :output-to "resources/public/js/app.js"
                                   :output-dir "resources/public/js/out"
                                   :source-map true
                                   :optimizations :none} }]})
```

```clojure
;; example/brepl.cljs
(ns example.brepl
  (:require [cljs.debux :refer-macros [dbg clog break]]
            [weasel.repl :as ws-repl] ))

(ws-repl/connect "ws://localhost:9001")

(def m {:e {:e 4, :d 3, :c 2, :b 1, :a 0},
        :d {:e 4, :d 3, :c 2, :b 1, :a 0},
        :c {:e 4, :d 3, :c 2, :b 1, :a 0},
        :b {:e 4, :d 3, :c 2, :b 1, :a 0},
        :a {:e 4, :d 3, :c 2, :b 1, :a 0}})

;; The evaluated value will be printed on browser REPL window.
(dbg m)

;; You can use clog macro in browser REPL as well.
;; The evaluated value will be printed on the console window
;; of the browser's developer tools.
(clog m)
```


## License
Copyright © 2015 Young Tae Kim

Distributed under the Eclipse Public License either version 1.0 or any later version.
