(ns recife.core
  (:require
   [alandipert.interpol8 :as int]
   [babashka.process :as p]
   [clojure.edn :as edn]
   [clojure.java.io :as io]
   [clojure.pprint :as pp]
   [clojure.repl :as repl]
   [clojure.set :as set]
   [clojure.string :as str]
   [clojure.walk :as walk]
   [cognitect.transit :as t]
   [lambdaisland.deep-diff2 :as ddiff]
   [malli.core :as m]
   [malli.error :as me]
   [medley.core :as medley]
   [potemkin :refer [def-map-type]]
   [recife.buffer :as r.buf]
   [recife.schema :as schema]
   [recife.trace :as rt]
   [recife.util :as u :refer [p*]]
   [taoensso.tufte :as tufte :refer [defnp p defnp-]]
   [tla-edn-2.core :as tla-edn]
   [tla-edn.spec :as spec]
   [recife.protocol :as proto]
   recife.class.recife-edn-value
   recife.records)
  (:import
   (java.io File)
   (lambdaisland.deep_diff2.diff_impl Mismatch Deletion Insertion)
   (tlc2.output IMessagePrinterRecorder MP EC)
   (tlc2.value.impl Value StringValue ModelValue RecordValue FcnRcdValue IntValue
                    TupleValue SetEnumValue BoolValue IntervalValue)
   (util UniqueString)
   (recife RecifeEdnValue)
   (recife.records RecifeProc)))

(set! *warn-on-reflection* false)
#_(set! *warn-on-reflection* true)

(set! *unchecked-math* false)

(def ^{:arglists (:arglists (meta #'r.buf/save!))}
  save!
  "Save data that can be fetched in real time, it can be used for statistics."
  r.buf/save!)

(defn save-and-flush!
  [bucket value]
  (try
    (r.buf/-save! [bucket value])
    (finally (r.buf/flush!))))

(def ^{:arglists (:arglists (meta #'r.buf/read-contents))}
  read-saved-data
  r.buf/read-contents)

(defn read-saved-data-if-new-content
  "Reads saved data if there is new content, otherwise, returns `nil`.
  See `save!`."
  []
  (when (r.buf/has-new-contents?)
    (r.buf/read-contents)))

;; Pre-compile regex patterns for performance
(def ^:private dot-pattern (re-pattern "\\."))
(def ^:private triple-underscore-pattern (re-pattern "___"))

;; Caches for munge/demunge operations
(defonce ^:private munge-cache (atom {}))
(defonce ^:private demunge-cache (atom {}))

(defn- custom-munge
  [v]
  (let [v-str (str v)]
    (or (get @munge-cache v-str)
        (let [result (str/replace (munge v) dot-pattern "___")]
          (swap! munge-cache assoc v-str result)
          result))))

(defn- to-edn-string
  [v]
  (let [s (str v)]
    (or (get @demunge-cache s)
        (let [replaced (str/replace s triple-underscore-pattern ".")
              result (keyword (repl/demunge replaced))]
          (swap! demunge-cache assoc s result)
          result))))

(defn- parse-tlc-isolated-output
  "Parse TLC stdout/stderr to extract results.
   Returns a result map compatible with tlc-result-handler output."
  [{:keys [exit-code output error]}]
  (let [combined (str output "\n" error)
        ;; Parse state counts
        distinct-states (some->> (re-find #"(\d+) distinct states? found" combined)
                                 second
                                 parse-long)
        generated-states (some->> (re-find #"(\d+) states? generated" combined)
                                  second
                                  parse-long)
        ;; Check for violations
        invariant-match (re-find #"Invariant\s+(\S+)\s+is violated" combined)
        action-prop-match (re-find #"Action property\s+(\S+)\s+is violated" combined)
        temporal-match (re-find #"Temporal properties were violated" combined)
        deadlock-match (re-find #"Deadlock reached" combined)
        stuttering-match (re-find #"stuttering" combined)
        back-to-state-match (re-find #"Back to state" combined)
        ;; Build violation info
        violation (cond
                    invariant-match
                    {:type :invariant
                     :name (to-edn-string (str/replace (second invariant-match) #"_COLON_" ""))}

                    action-prop-match
                    {:type :action-property
                     :name (to-edn-string (str/replace (second action-prop-match) #"_COLON_" ""))}

                    (and temporal-match stuttering-match)
                    {:type :stuttering}

                    (and temporal-match back-to-state-match)
                    {:type :back-to-state}

                    deadlock-match
                    {:type :deadlock}

                    :else nil)
        ;; Determine trace status
        trace (cond
                violation :violation-detected
                (zero? exit-code) :ok
                :else :error)]
    {:trace trace
     :trace-info (when violation {:violation violation})
     :distinct-states distinct-states
     :generated-states generated-states
     :isolated true
     :tlc-output output
     :tlc-error error}))

(defonce ^:private string-cache (atom {}))
(defonce ^:private keyword-cache (atom {}))
(defonce ^:private cache (atom {}))
(defonce ^:private values-cache (atom {}))
(defonce ^:private keys-cache (atom {}))

(defn- record-info-from-record
  [record]
  (let [names (p* ::record-info-from-record--to-edn
                  (mapv tla-edn/to-edn (.-names ^RecordValue record)))
        ;; Pre-compute index map for O(1) lookups instead of linear search
        index-map (into {} (map-indexed (fn [i k] [k i]) names))]
    {:names names
     :index-map index-map}))

(declare build-record-map)

;; TlaRecordMap uses an index-map for O(1) key lookups instead of linear search
(def-map-type TlaRecordMap [record record-info]
  (get [_ k default-value]
       (p* ::tla-record-map--get
           ;; Use pre-computed index-map for O(1) lookup instead of linear search
           (if-let [idx (get (:index-map record-info) k)]
             (tla-edn/to-edn
              (aget ^{:tag "[Ltlc2.value.impl.Value;"} (.-values ^RecordValue record)
                    idx))
             default-value)))

  (assoc [this k v]
         (p* ::tla-record-map--assoc-1
             (let [^{:tag "[Lutil.UniqueString;"}
                   names (.-names ^RecordValue record)
                   ^{:tag "[Ltlc2.value.impl.Value;"}
                   values (.-values ^RecordValue record)
                   length (alength names)
                   new-value (tla-edn/to-tla-value v)]
               ;; Use index-map for O(1) key existence check
               (if-let [existing-idx (get (:index-map record-info) k)]
                 ;; Key exists - update value at that index
                 (let [^{:tag "[Lutil.UniqueString;"}
                       new-names (make-array UniqueString length)
                       ^{:tag "[Ltlc2.value.impl.Value;"}
                       new-values (make-array Value length)]
                   (dotimes [i length]
                     (aset new-names i (aget names i))
                     (aset new-values i (if (= i existing-idx)
                                          new-value
                                          (aget values i))))
                   (build-record-map new-names new-values))
                 ;; Key doesn't exist - add new key-value pair
                 (let [val-tla (or (get @keyword-cache k)
                                   (let [result (UniqueString/of (custom-munge (symbol k)))]
                                     (swap! keyword-cache assoc k result)
                                     result))]
                   (build-record-map
                    (conj (into [] names) val-tla)
                    (conj (into [] values) new-value)))))))

  (dissoc [_ k]
          (p* ::tla-record-map--dissoc
              (let [names (transient [])
                    values (transient [])]
                (loop [record-names (.-names ^RecordValue record)
                       record-values (.-values ^RecordValue record)]
                  (let [n (first record-names)]
                    (when n
                      (when-not (= (tla-edn/to-edn n) k)
                        (conj! names n)
                        (conj! values (first record-values)))
                      (recur (rest record-names) (rest record-values)))))
                (build-record-map (persistent! names)
                                  (persistent! values)))))

  (keys [_]
        (p* ::tla-record-map--keys
            (:names record-info)))

  (empty [_]
         (p* ::tla-record-map--empty
             (build-record-map [] []))))

(defn- build-record-map
  ([record]
   (TlaRecordMap. record (record-info-from-record record)))
  ([names values]
   (let [record (RecordValue.
                 (tla-edn/typed-array UniqueString names)
                 (tla-edn/typed-array Value values)
                 false)]
     (TlaRecordMap. record (record-info-from-record record)))))

(defn- record-keys
  [v]
  (mapv (fn [name]
          (let [name-str (str name)]
            (or (get @keys-cache name-str)
                (let [result (tla-edn/-to-edn name)]
                  (swap! keys-cache assoc name-str result)
                  result))))
        (.-names ^RecordValue v)))

(defn- record-values
  [v]
  (mapv (fn [val]
          (let [val-str (str val)]
            (or (get @values-cache val-str)
                (let [result (tla-edn/-to-edn val)]
                  (swap! values-cache assoc val-str result)
                  result))))
        (.-values ^RecordValue v)))

(defmacro use-cache
  [v op]
  `(let [v# ~v]
     (or (get @cache v#)
         (let [result# ~op]
           (swap! cache assoc v# result#)
           result#))))

(defn- class-method
  [^Class klass ^String method-name]
  (let [m (.. klass (getDeclaredMethod method-name nil))]
    (.. m (setAccessible true))
    m))

(def ^:private fcn-rcd-is-tuple-method
  (class-method FcnRcdValue "isTuple"))

(defn ^:private fcn-rcd-is-tuple?
  [v]
  (boolean (.invoke ^java.lang.reflect.Method fcn-rcd-is-tuple-method v nil)))

(defn- private-field
  ([^Object obj ^String fn-name-string]
   (let [m (.. obj getClass (getDeclaredField fn-name-string))]
     (.. m (setAccessible true))
     (.. m (get obj))))
  ([^Object obj ^Class super-klass ^String fn-name-string]
   (let [m (.. super-klass (getDeclaredField fn-name-string))]
     (.. m (setAccessible true))
     (.. m (get obj)))))

(defrecord RecifeIntervalValue [low high])

(extend-protocol tla-edn/TLAPlusEdn
  RecordValue
  (-to-edn [v]
    (p* ::tla-edn--record
        (let [names (.-names ^RecordValue v)
              values (.-values ^RecordValue v)
              n (alength names)]
          (if (zero? n)
            {}
            (let [result (loop [i 0, acc (transient {})]
                           (if (< i n)
                             (recur (unchecked-inc i)
                                    (assoc! acc
                                            (tla-edn/-to-edn (aget names i))
                                            (tla-edn/-to-edn (aget values i))))
                             (persistent! acc)))]
              (if (= result {:tla-edn.record/empty? true})
                {}
                result))))))

  IntValue
  (-to-edn [v]
    (p* ::tla-edn--int
        (.val v)))

  StringValue
  (-to-edn [v']
    (p* ::tla-edn--string
        (let [v (.val v')]
          (or (get @string-cache v)
              (let [s (str/replace (str v) #"___" ".")
                    k (keyword (repl/demunge s))
                    result (if (= k :recife/null)
                             nil
                             k)]
                (swap! string-cache assoc v result)
                result)))))

  UniqueString
  (-to-edn [v]
    (p* ::tla-edn--unique-string
        (or (get @string-cache v)
            (let [result (to-edn-string v)]
              (swap! string-cache assoc v result)
              result))))

  BoolValue
  (-to-edn [v]
    (p* ::tla-edn--bool
        (.getVal v)))

  FcnRcdValue
  (-to-edn [v]
    (p* ::tla-edn--fcn
        (let [values (.-values v)
              n (alength values)]
          (if (fcn-rcd-is-tuple? v)
            ;; Tuple case: single-pass vector construction
            (loop [i 0, acc (transient [])]
              (if (< i n)
                (recur (unchecked-inc i) (conj! acc (tla-edn/-to-edn (aget values i))))
                (persistent! acc)))
            ;; Map case: single-pass map construction
            (let [domain (.-domain v)]
              (loop [i 0, acc (transient {})]
                (if (< i n)
                  (recur (unchecked-inc i)
                         (assoc! acc
                                 (tla-edn/-to-edn (aget domain i))
                                 (tla-edn/-to-edn (aget values i))))
                  (persistent! acc))))))))

  RecifeEdnValue
  (-to-edn [v]
    (p* ::tla-edn--recife-value
        (.-state v)))

  IntervalValue
  (-to-edn [v]
    (p* ::tla-edn--interval-value
        (RecifeIntervalValue. (.-low v) (.-high v))))

  SetEnumValue
  (-to-edn [v]
    (p* ::tla-edn--set
        (let [arr (.toArray (.-elems v))
              n (alength arr)]
          (loop [i 0, acc (transient #{})]
            (if (< i n)
              (recur (unchecked-inc i) (conj! acc (tla-edn/-to-edn (aget arr i))))
              (persistent! acc))))))

  TupleValue
  (-to-edn [v]
    (p* ::tla-edn--tuple
        (let [elems (.getElems v)
              n (alength elems)]
          (loop [i 0, acc (transient [])]
            (if (< i n)
              (recur (unchecked-inc i) (conj! acc (tla-edn/-to-edn (aget elems i))))
              (persistent! acc)))))))

(extend-protocol tla-edn/EdnToTla
  RecifeIntervalValue
  (tla-edn/-to-tla-value
    [v]
    (p* ::to-tla--interval-value
        (IntervalValue. (:low v) (:high v))))

  clojure.lang.Keyword
  (tla-edn/-to-tla-value
    [v]
    (p* ::to-tla--keyword
        (RecifeEdnValue. v)
        #_(use-cache v (RecifeEdnValue. v))
        #_(or (get @cache v)
              (let [result (RecifeEdnValue. v)]
                (swap! cache assoc v result)
                result)))
    #_(StringValue. ^String (custom-munge (symbol v))))

  nil
  (tla-edn/-to-tla-value [_]
    (p* ::to-tla--null
        (RecifeEdnValue. nil)
        #_(tla-edn/to-tla-value :recife/null)))

  clojure.lang.Seqable
  (tla-edn/-to-tla-value [v]
    (p* ::to-tla--seqable
        (tla-edn/to-tla-value (into [] v))))

  clojure.lang.Symbol
  (tla-edn/-to-tla-value [v]
    (p* ::to-tla--symbol
        (ModelValue/make (str v))))

  TlaRecordMap
  (tla-edn/-to-tla-value [v]
    (p* ::to-tla--tla-record-map
        #_(.record v)
        (RecifeEdnValue. v)))

  clojure.lang.Ratio
  (tla-edn/-to-tla-value [v]
    (p* ::to-tla--ratio
        (RecifeEdnValue. v)))

  BigInteger
  (tla-edn/-to-tla-value [v]
    (p* ::to-tla--big-int
        (IntValue/gen v)))

  clojure.lang.APersistentMap
  (-to-tla-value [coll]
    (p* ::to-tla--map
        (RecifeEdnValue. coll)
        #_(cond
            (empty? coll)
            (tla-edn/-to-tla-value {:tla-edn.record/empty? true})

            (every? keyword? (keys coll))
            (RecordValue.
             (tla-edn/typed-array UniqueString (mapv #(-> % key ^StringValue tla-edn/to-tla-value .getVal) coll))
             (tla-edn/typed-array Value (mapv #(-> % val tla-edn/to-tla-value) coll))
             false)

            :else
            (FcnRcdValue.
             (tla-edn/typed-array Value (mapv #(-> % key tla-edn/to-tla-value) coll))
             (tla-edn/typed-array Value (mapv #(-> % val tla-edn/to-tla-value) coll))
             false))))

  clojure.lang.PersistentVector
  (-to-tla-value [coll]
    (p* ::to-tla--vector
        (let [n (count coll)
              arr (make-array Value n)]
          (loop [i 0, s (seq coll)]
            (when s
              (aset arr i (tla-edn/-to-tla-value (first s)))
              (recur (unchecked-inc i) (next s))))
          (TupleValue. arr))))

  clojure.lang.PersistentList
  (-to-tla-value [coll]
    (p* ::to-tla--list
        (let [n (count coll)
              arr (make-array Value n)]
          (loop [i 0, s (seq coll)]
            (when s
              (aset arr i (tla-edn/-to-tla-value (first s)))
              (recur (unchecked-inc i) (next s))))
          (TupleValue. arr))))

  clojure.lang.PersistentHashSet
  (-to-tla-value [coll]
    (p* ::to-tla--hash-set
        (let [n (count coll)
              arr (make-array Value n)]
          (loop [i 0, s (seq coll)]
            (when s
              (aset arr i (tla-edn/-to-tla-value (first s)))
              (recur (unchecked-inc i) (next s))))
          (SetEnumValue. arr false))))

  String
  (-to-tla-value [v]
    (p* ::to-tla--string
        (throw (ex-info "We don't support strings for now, use a keyword instead."
                        {:string v}))
        #_(StringValue. v)))

  Boolean
  (-to-tla-value [v]
    (p* ::to-tla--boolean
        #_(RecifeEdnValue. v)
        (BoolValue. v)))

  Integer
  (-to-tla-value [v]
    (p* ::to-tla--integer
        (IntValue/gen v)))

  Long
  (-to-tla-value [v]
    (p* ::to-tla--long
        (IntValue/gen v))))

;; Single-pass key splitting for performance - avoids two passes through the map
(defn- split-by-namespace
  "Splits a map into [namespaced-keys, non-namespaced-keys] in a single pass."
  [m]
  (loop [entries (seq m)
         global (transient {})
         local (transient {})]
    (if entries
      (let [[k v] (first entries)]
        (if (namespace k)
          (recur (next entries) (assoc! global k v) local)
          (recur (next entries) global (assoc! local k v))))
      [(persistent! global) (persistent! local)])))

;; TODO: For serialized objecs, make it a custom random filename
;; so exceptions from concurrent executions do not mess with each other.
(def ^:private exception-filename
  ".recife-exception.ser")

(def ^:private invariant-data-filename
  ".recife-invariant-data.ser")

(def ^:private Lock (Object.))

(defn- serialize-obj [object ^String filename]
  (p* ::serialize-obj
      (locking Lock
        (when-not (.exists (io/as-file filename))
          (with-open [outp (-> (java.io.File. filename) java.io.FileOutputStream. java.io.ObjectOutputStream.)]
            (.writeObject outp object))))))

(defn- deserialize-obj [^String filename]
  (p* ::deserialize-obj
      (with-open [inp (-> (java.io.File. filename) java.io.FileInputStream. java.io.ObjectInputStream.)]
        (.readObject inp))))

(defn- process-operator*
  "Created so we can use the same \"meat\" when invoking the function
  in a context outside TLC."
  [identifier f self main-var]
  (p* ::process-operator*
      (let [global (dissoc main-var ::extra-args)
            #_#__ (when (::extra-args main-var)
                    (throw (ex-info (str "po* -- " self " ==== " (::extra-args main-var) " ==== " (keys main-var)) {})))
            local (get-in main-var [::procs self])
            extra-args (::extra-args main-var)
            ;; Use conj/into instead of multiple merge calls for better performance
            result (p ::result (f (p* ::result--merge
                                      (-> {:self self}
                                          (into global)
                                          (into extra-args)
                                          (into local)))))
            [result-global result-local] (split-by-namespace result)
            metadata (if (:recife/metadata result-global)
                       ;; There is some bug somewhere which prevent us of the form
                       ;;    Caused by java.lang.ClassCastException
                       ;;    clojure.lang.KeywordLookupSite$1 incompatible with
                       ;;    clojure.lang.IPersistentCollection
                       ;; which is triggered by multiple threads only, so we do this
                       ;; check here to try to prevent it.
                       (merge (:recife/metadata result-global)
                              {:context [identifier (merge {:self self} extra-args)]})
                       {:context [identifier (merge {:self self} extra-args)]})]
        (p* ::process-operator*--return
            (if (nil? result)
              (dissoc (assoc main-var :recife/metadata metadata) ::extra-args)
              (merge
               ;; All namespaced keywords are global.
               (dissoc result-global :recife/metadata)
               ;; While unamespaced keywords are part of the
               ;; process as local variables.
               {::procs (merge (::procs result-global)
                               {self
                                ;; Remove `extra-args` and `self`
                                ;; so they don't become stateful.
                                (apply dissoc result-local :self (keys extra-args))})
                :recife/metadata metadata}))))))

(defn process-operator
  [identifier f self-tla ^RecordValue main-var-tla ^Value extra-args]
  (p* ::process-operator
      (try
        (let [self (p ::process-operator--self
                      (tla-edn/to-edn self-tla))
              main-var (p ::process-operator--main-var
                          (tla-edn/to-edn main-var-tla))
              extra-args (p ::process-operator--extra-args
                            (tla-edn/to-edn extra-args))
              result (p ::process-operator--result
                        (process-operator* identifier f self
                                           (if (set? extra-args)
                                             main-var
                                             (assoc main-var ::extra-args extra-args))))]
          (p ::process-operator--to-tla-value
             (tla-edn/to-tla-value result)))
        (catch Exception e
          (serialize-obj e exception-filename)
          (throw e)))))

(defn- process-operator-local*
  [f self main-var]
  (p* ::deserialize-obj
      (let [global main-var
            local (get-in main-var [::procs self])]
        (f (merge {:self self} global local)))))

(defn- process-operator-local
  [f self-tla ^Value main-var-tla]
  (try
    (let [self (tla-edn/to-edn self-tla)
          main-var (tla-edn/to-edn main-var-tla)
          result (process-operator-local* f self main-var)]
      (tla-edn/to-tla-value result))
    (catch Exception e
      (serialize-obj e exception-filename)
      (throw e))))

(defn process-config-operator
  [f ^Value main-var-tla]
  (p* ::process-config-operator
      (try
        (let [main-var (p* ::process-config-operator--main-var
                           (tla-edn/to-edn main-var-tla))
              result (p* ::process-config-operator--result
                         (f main-var))]
          (if (vector? result)
            ;; If we have a vector, the first element is a boolean  and the
            ;; second one is data which will be appended to the result if
            ;; the boolean was falsy.
            (do (when-not (first result)
                  ;; Serialize data to be used later when building the result.
                  (serialize-obj (second result) invariant-data-filename))
                (tla-edn/to-tla-value (boolean (first result))))
            (tla-edn/to-tla-value (boolean result))))
        (catch Exception e
          (serialize-obj e exception-filename)
          (throw e)))))

(defn process-config-operator--primed
  [f ^Value main-var-tla ^Value main-var-tla']
  (p* ::process-config-operator--primed
      (try
        (let [main-var (p* ::process-config-operator--primed-main-var
                           (tla-edn/to-edn main-var-tla))
              main-var' (p* ::process-config-operator--primed-main-var
                            (tla-edn/to-edn main-var-tla'))
              result (p* ::process-config-operator--primed-result
                         (f main-var main-var'))]
          (if (vector? result)
            ;; If we have a vector, the first element is a boolean  and the
            ;; second one is data which will be appended to the result if
            ;; the boolean was falsy.
            (do (when-not (first result)
                  ;; Serialize data to be used later when building the result.
                  (serialize-obj (second result) invariant-data-filename))
                (tla-edn/to-tla-value (boolean (first result))))
            (tla-edn/to-tla-value (boolean result))))
        (catch Exception e
          (serialize-obj e exception-filename)
          (throw e)))))

(defn process-local-operator
  ([f ^Value main-var-tla]
   (process-local-operator f main-var-tla nil))
  ([f ^Value main-var-tla ^Value extra-args-tla]
   (p* ::process-local-operator
       (try
         (let [main-var (p ::local-op-to-edn-main-var-tla (tla-edn/to-edn main-var-tla))
               result (p ::local-op-result (f (if extra-args-tla
                                                (merge main-var (tla-edn/to-edn extra-args-tla))
                                                main-var)))]
           (p ::local-op-to-tla (tla-edn/to-tla-value result)))
         (catch Exception e
           (serialize-obj e exception-filename)
           (throw e))))))

(defn process-local-operator--primed
  [f ^Value main-var-tla ^Value main-var-tla']
  (p* ::process-local-operator--primed
      (try
        (let [main-var (p ::local-op-to-edn-main-var-tla (tla-edn/to-edn main-var-tla))
              main-var' (p ::local-op-to-edn-main-var-tla' (tla-edn/to-edn main-var-tla'))
              result (p ::local-op-result (f main-var main-var'))]
          (p ::local-op-to-tla (tla-edn/to-tla-value result)))
        (catch Exception e
          (serialize-obj e exception-filename)
          (throw e)))))

(defn parse
  "`f` is a function which receives one argument, a vector, the
  first element is the `result` and the last one is the `expr`."
  ([expr]
   (parse expr first))
  ([expr f]
   (let [result (cond
                  (vector? expr)
                  (let [[id & args] expr]
                    (case id
                      :exists
                      (format "\\E %s : (%s)"
                              (->> (first args)
                                   (mapv (fn [[k v]]
                                           (format "%s \\in (%s)"
                                                   (parse (custom-munge k) f)
                                                   (parse v f))))
                                   (str/join ", "))
                              (parse (last args) f))

                      :for-all
                      (format "\\A %s : (%s)"
                              (->> (first args)
                                   (mapv (fn [[k v]]
                                           (format "%s \\in (%s)"
                                                   (parse (custom-munge k) f)
                                                   (parse v f))))
                                   (str/join ", "))
                              (parse (last args) f))

                      :eventually
                      (format "<>(%s)" (parse (first args) f))

                      :box
                      (format "[][%s]_main_var" (parse (first args) f))

                      :always
                      (format "[](%s)" (parse (first args) f))

                      :leads-to
                      (format "(%s) ~> (%s)"
                              (parse (first args) f)
                              (parse (second args) f))

                      :implies
                      (format "(%s) => (%s)"
                              (parse (first args) f)
                              (parse (second args) f))

                      :not
                      (format "~(%s)"
                              (parse (first args) f))

                      :=
                      (format "(%s) = (%s)"
                              (parse (first args) f)
                              (parse (second args) f))

                      :current-state
                      (format "main_var")

                      :next-state
                      (format "main_var'")

                      :invoke
                      (parse {:env (->> (first args)
                                        (mapv (fn [[k v]]
                                                [k (symbol (custom-munge v))]))
                                        (into {}))
                              :fn (last args)}
                             f)

                      :fair
                      (format "WF_main_var_seq(%s)"
                              (parse (first args) f))

                      :fair+
                      (format "SF_main_var_seq(%s)"
                              (parse (first args) f))

                      :call
                      (format "%s(%s%s)"
                              (custom-munge (first args))
                              (parse (second args))
                              (if-let [call-args (seq (drop 2 args))]
                                (str ", "
                                     (->> call-args
                                          (mapv #(parse % f))
                                          (str/join ", ")))
                                ""))

                      :or
                      (->> args
                           (mapv (comp #(str "(" % ")") #(parse % f)))
                           (str/join " \\/ "))

                      :and
                      (->> args
                           (mapv (comp #(str "(" % ")") #(parse % f)))
                           (str/join " /\\ "))

                      :if
                      (format "IF (%s) THEN (%s) ELSE (%s)"
                              (parse (first args) f)
                              (parse (second args) f)
                              (parse (last args) f))

                      :raw
                      (first args)

                      (format "%s(%s)"
                              (symbol (custom-munge id))
                              (->> (conj (vec args) "main_var")
                                   (str/join ", ")))))

                  (keyword? expr)
                  (tla-edn/to-tla-value expr)

                  (set? expr)
                  (tla-edn/to-tla-value expr)

                  (sequential? expr)
                  (tla-edn/to-tla-value (set expr))

                  :else
                  expr)]
     (f [result expr]))))

(defn tla
  [identifier expr]
  (let [form (parse expr)]
    {:identifier (str (symbol (custom-munge identifier))
                      (some->> (:receives expr)
                               (str/join ", ")
                               (format "(%s, _main_var)")))
     :form form
     :recife.operator/type :tla-only}))

(defn tla-repl
  [tla-expr]
  (let [tmp (java.nio.file.Files/createTempDirectory "repltest"
                                                     (into-array java.nio.file.attribute.FileAttribute []))]
    (-> (tlc2.REPL. tmp)
        (.processValue tla-expr)
        tla-edn/to-edn)))

#_(tla-repl "RandomSubset(2, [1..43 -> {TRUE, FALSE}])")
#_(tla-repl "RandomSubset(2, [2..5 -> {TRUE, FALSE}])")
#_(count (tla-repl "[1..43 -> {TRUE, FALSE}]"))
#_(tla-repl "CHOOSE x \\in RandomSubset(2, [2..43 -> {TRUE, FALSE}]): TRUE")

(defn context-from-state
  [state]
  (if (vector? state)
    (get-in state [1 :recife/metadata :context])
    (get-in state [:recife/metadata :context])))

(defmulti simulate
  "Simulate trace."
  (fn [context _state]
    (first context)))

(defmulti operator-local
  "Used for helper functions (e.g. to help with non-determinism).
  Instead of creating a new operator, we create a unique one,
  `recife-operator-local`, and dispatch it based on `:step` and
  `:key` fields."
  (fn [args]
    (select-keys args [:step :key])))

(spec/defop recife_operator_local {:module "spec"}
  [self ^Value params main-var]
  (p* ::recife_operator_local
      (let [{:keys [:f]} (operator-local (tla-edn/to-edn params))]
        (process-operator-local f self main-var))))

(spec/defop recife_check_extra_args {:module "spec"}
  [^Value main-var]
  (p* ::recife_check_extra_args
      #_(when (contains? (set (keys (tla-edn/to-edn main-var)))
                         ::extra-args)

          (throw (ex-info (str "extra --- " (tla-edn/to-tla-value
                                             (contains? (set (keys (tla-edn/to-edn main-var)))
                                                        ::extra-args))
                               " ==== "
                               (::extra-args (tla-edn/to-edn main-var))) {})))
      (tla-edn/to-tla-value
       (contains? (set (keys (tla-edn/to-edn main-var)))
                  ::extra-args))))

(spec/defop recife_check_inequality {:module "spec"}
  [^Value main-var ^Value main-var']
  (p* ::recife_check_inequality
      (tla-edn/to-tla-value
       (not= (dissoc (tla-edn/to-edn main-var) :recife/metadata)
             (dissoc (tla-edn/to-edn main-var') :recife/metadata)))))

(spec/defop recife_my_view {:module "spec"}
  [^Value main-var]
  (p* ::recife_my_view
      (tla-edn/to-tla-value (dissoc (tla-edn/to-edn main-var) :recife/metadata))))

;; \\A self \\in DOMAIN main_var[procs]: main_var[procs][self][\"pc\"] = done
(spec/defop recife_check_done {:module "spec"}
  [^Value main-var]
  (p* ::recife_check_done
      (tla-edn/to-tla-value
       (->> (::procs (tla-edn/to-edn main-var))
            vals
            (every? #(= (:pc %) ::done))))))

(spec/defop recife_check_pc {:module "spec"}
  [^Value main-var ^Value self ^Value identifier]
  #_[^Value main-var ^StringValue self ^StringValue identifier]
  (p* ::recife_check_pc
      (tla-edn/to-tla-value
       (= (get-in (tla-edn/to-edn main-var)
                  [::procs (tla-edn/to-edn self) :pc])
          (tla-edn/to-edn identifier)))))

(declare temporal-property)

(defn reg
  ([identifier expr]
   (reg identifier {} expr))
  ([identifier opts expr]
   (let [op (eval
             `(spec/defop ~(symbol (str (custom-munge identifier) "2")) {:module "spec"}
                [~'self ~'main-var ~'extra-args]
                (process-operator ~identifier ~expr ~'self ~'main-var ~'extra-args)))]
     (defmethod simulate identifier
       [context state]
       (let [self (get-in context [1 :self])
             identifier (first context)
             context' (second context)]
         ;; This is the same function that we use at `process-operator`.
         (process-operator* identifier expr self (assoc (if (vector? state)
                                                          (second state)
                                                          state)
                                                        ::extra-args context'))))
     (with-meta
       {:identifier (str (symbol (str (custom-munge identifier) "2"))
                         "(self, _main_var, _extra_args) == self = self /\\ _main_var = _main_var /\\ _extra_args = _extra_args\n\n"
                         (symbol (custom-munge identifier))
                         "(self, _main_var)")
        :op-ns (-> op meta :op-ns)
        :recife.operator/name (str (symbol (custom-munge identifier)))
        :recife/fairness (cond
                           (or (-> expr meta :fair)
                               (-> opts meta :fair)) :recife.fairness/weakly-fair
                           (or (-> expr meta :fair+)
                               (-> opts meta :fair+)) :recife.fairness/strongly-fair)
        :recife/non-deterministic-params (set (keys opts))
        :recife/fairness-map (some->> (or (-> expr meta :fairness) (-> opts meta :fairness))
                                      (temporal-property (keyword (str (symbol identifier) "-fairness"))))
        :expr expr
        :form (parse [:if [:raw "recife_check_extra_args(_main_var)"]
                      [:and
                       [:raw (format "recife_check_pc(_main_var, self, %s)"
                                     (str (tla-edn/to-tla-value identifier)))]
                       [:raw (str "main_var' = "
                                  (symbol (str (custom-munge identifier) "2"))
                                  "(self, _main_var, {})")]
                       [:raw "recife_check_inequality(_main_var, main_var')"]]
                      [:and
                       [:raw (format "recife_check_pc(_main_var, self, %s)"
                                     (str (tla-edn/to-tla-value identifier)))]
                       (if (seq opts)
                         [:raw (parse [:exists (->> opts
                                                    (mapv (fn [[k v]]
                                                            ;; Here we want to achieve non determinism.
                                                            [(symbol (str (custom-munge k)))
                                                             ;; If the value is empty, we return
                                                             ;; a set with a `nil` (`#{nil}`) so
                                                             ;; we don't have bogus deadlocks.
                                                             (cond
                                                               ;; If we have a coll here, then we want
                                                               ;; to get one of these hardcoded elements.
                                                               (coll? v)
                                                               (if (seq v)
                                                                 (parse (set v))
                                                                 #{nil})

                                                               ;; A keyword means that the user wants
                                                               ;; to use one of the global variables
                                                               ;; as a source of non determinism.
                                                               ;; TODO: Maybe if the user passes a
                                                               ;; unamespaced keyword we can use a
                                                               ;; local variable?
                                                               #_(keyword? v)
                                                               #_[:raw
                                                                  (let [mv (str " _main_var[" (parse v) "] ")]
                                                                    (format " IF %s = {} THEN %s ELSE %s"
                                                                            mv
                                                                            (parse #{nil})
                                                                            mv))]

                                                               ;; For functions, instead of creating a
                                                               ;; new operator, we use a defmethod used
                                                               ;; by a hardcoded operator (`recife-operator-local`).
                                                               (or (keyword? v)
                                                                   (fn? v))
                                                               (do (defmethod operator-local {:step identifier
                                                                                              :key k}
                                                                     [_]
                                                                     {:step identifier
                                                                      :key k
                                                                      :f (comp (fn [result]
                                                                                 (if (seq result)
                                                                                   (set result)
                                                                                   #{nil}))
                                                                               v)})
                                                                   [:raw (str " recife_operator_local"
                                                                              (format "(self, %s, _main_var)"
                                                                                      (tla-edn/to-tla-value
                                                                                       {:step identifier
                                                                                        :key k})))])

                                                               :else
                                                               (throw (ex-info "Unsupported type"
                                                                               {:step identifier
                                                                                :opts opts
                                                                                :value [k v]})))])))
                                       [:raw (str "\nmain_var' = "
                                                  (symbol (str (custom-munge identifier) "2"))
                                                  (format "(self, _main_var, %s)"
                                                          (->> (keys opts)
                                                               (mapv #(hash-map % (symbol (custom-munge %))))
                                                               (into {})
                                                               tla-edn/to-tla-value)))]])]
                         [:raw (str "main_var' = "
                                    (symbol (str (custom-munge identifier) "2"))
                                    "(self, _main_var, {})")])
                       ;; With it we can test deadlock.
                       [:raw "recife_check_inequality(_main_var, main_var')"]]])
        :recife.operator/type :operator}
       (merge (meta expr) (meta opts))))))

(defn invariant
  ([identifier expr]
   (invariant identifier nil expr))
  ([identifier doc-string expr]
   (let [op (eval
             `(spec/defop ~(symbol (str (custom-munge identifier) "2")) {:module "spec"}
                [^Value ~'main-var]
                (process-config-operator ~expr ~'main-var)))]
     {:identifier (symbol (custom-munge identifier))
      :op-ns (-> op meta :op-ns)
      :identifier-2 (str (symbol (str (custom-munge identifier) "2")) "(_main_var) == _main_var = _main_var")
      :form (str (symbol (str (custom-munge identifier) "2"))
                 "(main_var)")
      :recife.operator/type :invariant
      :doc-string doc-string})))

(defn state-constraint
  [identifier expr]
  (let [op (eval
            `(spec/defop ~(symbol (str (custom-munge identifier) "2")) {:module "spec"}
               [^Value ~'main-var]
               (process-config-operator ~expr ~'main-var)))]
    {:identifier (symbol (custom-munge identifier))
     :op-ns (-> op meta :op-ns)
     :identifier-2 (str (symbol (str (custom-munge identifier) "2")) "(_main_var) == _main_var = _main_var")
     :form (str (symbol (str (custom-munge identifier) "2"))
                "(main_var)")
     :recife.operator/type :state-constraint}))

(defn action-constraint
  "`expr` is a function of two arguments, the first one is the main var and the
  next one is the main var in the next state (primed)."
  [identifier expr]
  (let [op (eval
            `(spec/defop ~(symbol (str (custom-munge identifier) "2")) {:module "spec"}
               [^Value ~'main-var ^Value ~'main-var']
               (process-config-operator--primed ~expr ~'main-var ~'main-var')))]
    {:identifier (symbol (custom-munge identifier))
     :op-ns (-> op meta :op-ns)
     :identifier-2 (str (symbol (str (custom-munge identifier) "2")) "(_main_var, _main_var_2) == _main_var = _main_var /\\ _main_var_2 = _main_var_2")
     :form (str (symbol (str (custom-munge identifier) "2"))
                "(main_var, main_var')")
     :recife.operator/type :action-constraint}))

(defn- compile-temporal-property
  "If `:action?` is set, the function will receive 2 arguments, the current var
  and the next one."
  [collector identifier {:keys [action?]}]
  (let [        ;; With `counter` we are able to use many functions for the same
        ;; temporal property.
        counter (atom 100)]
    (fn [id-and-args]
      (parse id-and-args (fn [[result expr]]
                           (cond
                             (fn? expr)
                             (let [op (eval
                                       (if action?
                                         `(spec/defop ~(symbol (str (custom-munge identifier)
                                                                    @counter))
                                            {:module "spec"}
                                            [^Value ~'main-var ^Value ~'main-var']
                                            (process-local-operator--primed ~expr ~'main-var ~'main-var'))
                                         `(spec/defop ~(symbol (str (custom-munge identifier)
                                                                    @counter))
                                            {:module "spec"}
                                            [^Value ~'main-var]
                                            (process-local-operator ~expr ~'main-var))))]
                               (swap! collector conj {:identifier (str (symbol (str (custom-munge identifier)
                                                                                    @counter))
                                                                       (if action?
                                                                         "(_main_var, _main_var_2) == _main_var = _main_var /\\ _main_var_2 = _main_var_2"
                                                                         "(_main_var) == _main_var = _main_var"))
                                                      :op-ns (-> op meta :op-ns)})
                               (try
                                 (str (symbol (str (custom-munge identifier)
                                                   @counter))
                                      (if action?
                                        "(main_var, main_var')"
                                        "(main_var)"))
                                 (finally
                                   (swap! counter inc))))

                             ;; This means that we called invoke and we are calling a native
                             ;; function.
                             (map? expr)
                             (let [{:keys [:env] f :fn} expr
                                   op (eval
                                       (if action?
                                         `(spec/defop ~(symbol (str (custom-munge identifier)
                                                                    @counter))
                                            {:module "spec"}
                                            [^Value ~'main-var ^Value ~'main-var']
                                            (process-local-operator--primed ~f ~'main-var ~'main-var'))
                                         `(spec/defop ~(symbol (str (custom-munge identifier)
                                                                    @counter))
                                            {:module "spec"}
                                            [^Value ~'main-var ^Value ~'extra-args]
                                            (process-local-operator ~f ~'main-var ~'extra-args))))]
                               (swap! collector conj {:identifier (str (symbol (str (custom-munge identifier)
                                                                                    @counter))
                                                                       (if action?
                                                                         "(_main_var, _main_var_2) == _main_var = _main_var /\\ _main_var_2 = _main_var_2"
                                                                         "(_main_var, _extra_args) == _main_var = _main_var /\\ _extra_args = _extra_args"))
                                                      :op-ns (-> op meta :op-ns)})
                               (try
                                 (str (symbol (str (custom-munge identifier)
                                                   @counter))
                                      (if action?
                                        "(main_var, main_var')"
                                        (if (seq env)
                                          (format "(main_var, %s)"
                                                  (tla-edn/to-tla-value env))
                                          "(main_var, {})")))
                                 (finally
                                   (swap! counter inc))))

                             :else
                             result))))))

(defn temporal-property
  ([identifier expr]
   (temporal-property identifier expr {}))
  ([identifier expr {:keys [action?] :as opts}]
   (let [collector (atom [])
         form ((compile-temporal-property collector identifier opts) expr)]
     {:identifier (symbol (custom-munge identifier))
      :identifiers (mapv :identifier @collector)
      :op-nss (mapv :ns @collector)
      :form form
      :recife.operator/type :temporal-property})))

(defn fairness
  [identifier expr]
  (merge (temporal-property identifier expr)
         {:recife.operator/type :fairness}))

(defn goto
  [db identifier]
  (assoc db :pc identifier))

(defn done
  "Finishes step so deadlock is not triggered for this step."
  [db]
  (assoc db :pc ::done))

(defn all-done?
  [db]
  (every? #(= (:pc %) ::done)
          (-> db ::procs vals)))

(defmacro implies
  "Like `when`, but it returns `true` if `condition` is falsy."
  [condition & body]
  `(if ~condition
     ~@body
     true))

(defn one-of
  "Tells Recife to choose one of the values as its initial value."
  ([values]
   (one-of nil values))
  ([identifier values]
   {::type ::one-of
    ::possible-values values
    ::identifier (when-let [v (some->> identifier hash)]
                   (->> (Math/abs ^Integer v)
                        (str "G__")
                        symbol))}))

(defn- module-template
  [{:keys [:init :next :spec-name :operators]}]
  (let [collected-ranges (atom #{})
        formatted-init-expressions (-> (->> init
                                            (walk/prewalk (fn [v]
                                                            (if (= (::type v) ::one-of)
                                                              (let [identifier (or (::identifier v)
                                                                                   (gensym))
                                                                    v' (assoc v ::identifier identifier)]
                                                                (swap! collected-ranges conj v')
                                                                (keyword identifier))
                                                              v)))
                                            tla-edn/to-tla-value
                                            str)
                                       ;; Remove quote strings from generated symbols so we can
                                       ;; use in a TLA+ mapping later.
                                       (str/replace #"\"G__\d+\"" (fn [s] (subs s 1 (dec (count s))))))
        formatted-collected-ranges (->> @collected-ranges
                                        ;; Sort it so it's more deterministic.
                                        (sort-by ::identifier)
                                        (mapv (fn [{:keys [::identifier ::possible-values]}]
                                                (format "%s \\in {%s}"
                                                        identifier
                                                        (->> possible-values
                                                             (mapv tla-edn/to-tla-value)
                                                             (str/join ", ")))))
                                        (str/join " /\\ "))
        formatted-init (if (seq @collected-ranges)
                         (format "%s /\\ main_var = %s" formatted-collected-ranges formatted-init-expressions)
                         (format "main_var = %s" formatted-init-expressions))
        formatted-invariants (->> operators
                                  (filter (comp #{:invariant} :recife.operator/type))
                                  (mapv #(format "%s\n\n%s ==\n  %s" (:identifier-2 %) (:identifier %) (:form %)))
                                  (str/join "\n\n"))
        formatted-constraints (->> operators
                                   (filter (comp #{:state-constraint} :recife.operator/type))
                                   (mapv #(format "%s\n\n%s ==\n  %s" (:identifier-2 %) (:identifier %) (:form %)))
                                   (str/join "\n\n"))
        formatted-action-constraints (->> operators
                                          (filter (comp #{:action-constraint} :recife.operator/type))
                                          (mapv #(format "%s\n\n%s ==\n  %s" (:identifier-2 %) (:identifier %) (:form %)))
                                          (str/join "\n\n"))
        config-invariants (or (some->> operators
                                       (filter (comp #{:invariant} :recife.operator/type))
                                       seq
                                       (mapv #(format "  %s" (:identifier %)))
                                       (str/join "\n")
                                       (str "INVARIANTS\n"))
                              "")
        config-constraints (or (some->> operators
                                        (filter (comp #{:state-constraint} :recife.operator/type))
                                        seq
                                        (mapv #(format "  %s" (:identifier %)))
                                        (str/join "\n")
                                        (str "CONSTRAINT\n"))
                               "")
        config-action-constraints (or (some->> operators
                                               (filter (comp #{:action-constraint} :recife.operator/type))
                                               seq
                                               (mapv #(format "  %s" (:identifier %)))
                                               (str/join "\n")
                                               (str "ACTION_CONSTRAINT\n"))
                                      "")
        formatted-temporal-properties (->> operators
                                           (filter (comp #{:temporal-property} :recife.operator/type))
                                           (mapv #(format "%s\n\n%s ==\n  %s"
                                                          (->> (:identifiers %)
                                                               (str/join "\n\n"))
                                                          (:identifier %)
                                                          (:form %)))
                                           (str/join "\n\n"))
        config-temporal-properties (or (some->> operators
                                                (filter (comp #{:temporal-property} :recife.operator/type))
                                                seq
                                                (mapv #(format "  %s" (:identifier %)))
                                                (str/join "\n")
                                                (str "PROPERTY\n"))
                                       "")
        formatted-fairness-operators (or (some->> operators
                                                  (filter (comp #{:fairness} :recife.operator/type))
                                                  seq
                                                  (mapv #(format "%s\n\n%s ==\n  %s"
                                                                 (->> (:identifiers %)
                                                                      (str/join "\n\n"))
                                                                 (:identifier %)
                                                                 (:form %)))
                                                  (str/join "\n\n"))
                                         "")
        #_#__ (def operators operators)
        unchanged-helper-variables (or (some->> @collected-ranges
                                                seq
                                                (mapv (fn [{::keys [identifier]}] (format "%s' = %s" identifier identifier)))
                                                (str/join " /\\ ")
                                                (str " /\\ "))
                                       "")
        fairness-identifiers (or (some->> operators
                                          (filter (comp #{:fairness} :recife.operator/type))
                                          seq
                                          (mapv :identifier)
                                          (str/join " /\\ ")
                                          (str " /\\ "))
                                 "")
        formatted-operators (->> (concat (->> operators
                                              (filter (comp #{:operator} :recife.operator/type))
                                              (mapv #(format "%s ==\n  %s%s"
                                                             (:identifier %)
                                                             (:form %)
                                                             (if (seq unchanged-helper-variables)
                                                               (str " /\\ " "(" unchanged-helper-variables ")")
                                                               ""))))
                                         (->> operators
                                              (filter (comp #{:tla-only} :recife.operator/type))
                                              (mapv #(format "%s ==\n  %s"
                                                             (:identifier %)
                                                             (:form %)))))
                                 (str/join "\n\n"))
        ;; `terminating` prevents deadlock
        terminating  "(recife_check_done(main_var) /\\ UNCHANGED vars)"
        helper-variables (or (some->> @collected-ranges
                                      seq
                                      (mapv (fn [{::keys [identifier]}] (format "%s" identifier)))
                                      (str/join ", ")
                                      (str ", "))
                             "")
        fairness-operators (some->> operators
                                    (filter :recife/fairness)
                                    seq
                                    (mapv (juxt :recife/fairness :recife.operator/name))
                                    (mapv (fn [[fairness name]]
                                            [:raw (format "\\A self \\in %s: %s_main_var_seq(%s(self, main_var))"
                                                          (->> (keys (::procs init))
                                                               set
                                                               tla-edn/to-tla-value)
                                                          (case fairness
                                                            :recife.fairness/weakly-fair "WF"
                                                            :recife.fairness/strongly-fair "SF")
                                                          name)])))
        formatted-fairness (or
                            (some->> (concat fairness-operators
                                             (some->> operators
                                                      (filter :recife/fairness-map)
                                                      seq
                                                      (mapv (fn [{:keys [:recife/fairness-map]}]
                                                              [:raw (:form fairness-map)]))))
                                     seq
                                     (into [:and])
                                     parse)
                            "TRUE")
        other-identifiers (or (some->> operators
                                       (filter :recife/fairness-map)
                                       seq
                                       (mapcat (fn [{:keys [:recife/fairness-map]}]
                                                 (:identifiers fairness-map)))
                                       (str/join "\n\n"))
                              "")
        formatted-next-form (if (seq (:form next))
                              (:form next)
                              "FALSE")]
    (int/i "
-------------------------------- MODULE #{spec-name} --------------------------------

EXTENDS Integers, TLC

VARIABLES main_var #{helper-variables}

vars == << main_var #{helper-variables} >>

main_var_seq == << main_var >>

recife_operator_local(_self, _params, _main_var) == _self = _self /\\ _params = _params /\\ _main_var = _main_var

recife_check_extra_args(_main_var) == _main_var = main_var

recife_check_inequality(_main_var, _main_var_2) == _main_var = main_var /\\ _main_var_2 = _main_var_2

recife_my_view(_main_var) == _main_var = main_var

recife_check_done(_main_var) == _main_var = main_var

recife_check_pc(_main_var, self, identifier) == _main_var = main_var /\\ self = self /\\ identifier = identifier

#{other-identifiers}

__init ==
   #{formatted-init}

#{formatted-operators}

#{(:identifier next)} ==
   #{formatted-next-form}

Init == __init

Next == (#{(:identifier next)}) \\/ #{terminating}

#{formatted-invariants}

#{formatted-constraints}

#{formatted-action-constraints}

#{formatted-temporal-properties}

#{formatted-fairness-operators}

Fairness ==
   #{formatted-fairness}#{fairness-identifiers}

Spec == /\\ Init
        /\\ [][Next]_vars
        /\\ Fairness

MyView == << recife_my_view(main_var) >>

=============================================================================

-------------------------------- CONFIG #{spec-name} --------------------------------

SPECIFICATION
   Spec

VIEW
   MyView

#{config-constraints}

#{config-action-constraints}

#{config-invariants}

#{config-temporal-properties}

=============================================================================
")))

;; EdnStateWriter
(def ^:private edn-states-atom (atom {}))

(defn- edn-save-state
  [^tlc2.tool.TLCState tlc-state]
  (let [edn-state (->> tlc-state
                       .getVals
                       (some #(when (= (str (key %)) "main_var")
                                (val %)))
                       tla-edn/to-edn)]
    (swap! edn-states-atom assoc-in
           [:states (.fingerPrint tlc-state)]
           {:state (dissoc edn-state :recife/metadata)
            :successors #{}})))

(defn- edn-rank-state
  [^tlc2.tool.TLCState tlc-state]
  (swap! edn-states-atom update-in
         [:ranks (dec (.getLevel tlc-state))]
         (fnil conj #{}) (.fingerPrint tlc-state)))

(defn- edn-save-successor
  [^tlc2.tool.TLCState tlc-state ^tlc2.tool.TLCState tlc-successor]
  (let [successor-state (->> tlc-successor
                             .getVals
                             (some #(when (= (str (key %)) "main_var")
                                      (val %)))
                             tla-edn/to-edn)]
    (swap! edn-states-atom update-in
           [:states (.fingerPrint tlc-state) :successors]
           conj
           [(.fingerPrint tlc-successor) (get-in successor-state [:recife/metadata :context])])))

(defn- state-new?
  "Check if the state flags indicate this is a new (unseen) state.
  In the new TLC API, stateFlags is a short where 0 = unseen."
  [state-flags]
  (= (int state-flags) (int tlc2.util.IStateWriter/IsUnseen)))

(defn- edn-write-state
  [_state-writer state successor _action-checks _from _length state-flags _visualization _action]
  (when (state-new? state-flags)
    (edn-save-state successor))
  (edn-rank-state state)
  (edn-save-successor state successor))

#_(clojure.edn/read-string (slurp "edn-states-atom.edn"))

;; EdnStateWriter using proxy because Clojure doesn't support short type hints in defrecord
(defn make-edn-state-writer
  "Create an IStateWriter that stores state in edn-states-atom."
  []
  (proxy [tlc2.util.IStateWriter] []
    (writeState
      ([state]
       (edn-save-state state)
       (edn-rank-state state))
      ([state successor state-flags]
       (.writeState ^tlc2.util.IStateWriter this state successor state-flags
                    tlc2.util.IStateWriter$Visualization/DEFAULT))
      ([state successor state-flags action-or-viz]
       (if (instance? tlc2.util.IStateWriter$Visualization action-or-viz)
         ;; (state, successor, stateFlags, visualization)
         (edn-write-state this state successor nil 0 0 state-flags action-or-viz nil)
         ;; (state, successor, stateFlags, action)
         (edn-write-state this state successor nil 0 0 state-flags
                          tlc2.util.IStateWriter$Visualization/DEFAULT action-or-viz)))
      ([state successor state-flags action pred]
       ;; (state, successor, stateFlags, action, semanticNode)
       (edn-write-state this state successor nil 0 0 state-flags
                        tlc2.util.IStateWriter$Visualization/DEFAULT action))
      ([state successor action-checks from length state-flags]
       (edn-write-state this state successor action-checks from length state-flags
                        tlc2.util.IStateWriter$Visualization/DEFAULT nil))
      ([state successor action-checks from length state-flags visualization]
       (edn-write-state this state successor action-checks from length state-flags visualization nil)))
    (close []
      (spit "edn-states-atom.edn" @edn-states-atom))
    (getDumpFileName [] nil)
    (isNoop [] false)
    (isDot [] false)
    (isConstrained [] false)
    (snapshot [] nil)))

;; FileStateWriter
(defn- file-sw-save-state
  [{:keys [:edn-states-atom]} ^tlc2.tool.TLCState tlc-state]
  (let [edn-state (->> tlc-state
                       .getVals
                       (some #(when (= (str (key %)) "main_var")
                                (val %)))
                       tla-edn/to-edn)]
    (swap! edn-states-atom assoc-in
           [:states (.fingerPrint tlc-state)]
           {:state (dissoc edn-state :recife/metadata)
            :successors #{}})))

(defn- file-sw-rank-state
  [{:keys [:edn-states-atom]} ^tlc2.tool.TLCState tlc-state]
  (swap! edn-states-atom update-in
         [:ranks (dec (.getLevel tlc-state))]
         (fnil conj #{}) (.fingerPrint tlc-state)))

(defn- file-sw-save-successor
  [{:keys [:edn-states-atom]} ^tlc2.tool.TLCState tlc-state ^tlc2.tool.TLCState tlc-successor]
  (let [successor-state (->> tlc-successor
                             .getVals
                             (some #(when (= (str (key %)) "main_var")
                                      (val %)))
                             tla-edn/to-edn)]
    (swap! edn-states-atom update-in
           [:states (.fingerPrint tlc-state) :successors]
           conj
           [(.fingerPrint tlc-successor) (get-in successor-state [:recife/metadata :context])])))

(defn- file-sw-write-state
  [this state successor _action-checks _from _length state-flags visualization _action]
  ;; If it's stuttering, we don't put it as a successor.
  (when-not (= visualization tlc2.util.IStateWriter$Visualization/STUTTERING)
    (when (state-new? state-flags)
      (file-sw-save-state this successor))
    (file-sw-rank-state this state)
    (file-sw-save-successor this state successor)))

;; FileStateWriter using proxy because Clojure doesn't support short type hints in defrecord
(defn make-file-state-writer
  "Create an IStateWriter that stores state to a file using transit."
  [writer output-stream edn-states-atom file-path]
  (let [sw-state {:writer writer
                  :output-stream output-stream
                  :edn-states-atom edn-states-atom
                  :file-path file-path}]
    (proxy [tlc2.util.IStateWriter] []
      (writeState
        ([s]
         (file-sw-save-state sw-state s)
         (file-sw-rank-state sw-state s))
        ([s successor state-flags]
         (.writeState ^tlc2.util.IStateWriter this s successor state-flags
                      tlc2.util.IStateWriter$Visualization/DEFAULT))
        ([s successor state-flags action-or-viz]
         (if (instance? tlc2.util.IStateWriter$Visualization action-or-viz)
           ;; (state, successor, stateFlags, visualization)
           (file-sw-write-state sw-state s successor nil 0 0 state-flags action-or-viz nil)
           ;; (state, successor, stateFlags, action)
           (file-sw-write-state sw-state s successor nil 0 0 state-flags
                                tlc2.util.IStateWriter$Visualization/DEFAULT action-or-viz)))
        ([s successor state-flags action pred]
         ;; (state, successor, stateFlags, action, semanticNode)
         (file-sw-write-state sw-state s successor nil 0 0 state-flags
                              tlc2.util.IStateWriter$Visualization/DEFAULT action))
        ([s successor action-checks from length state-flags]
         (file-sw-write-state sw-state s successor action-checks from length state-flags
                              tlc2.util.IStateWriter$Visualization/DEFAULT nil))
        ([s successor action-checks from length state-flags visualization]
         (file-sw-write-state sw-state s successor action-checks from length state-flags visualization nil)))
      (close []
        (t/write writer @edn-states-atom)
        (reset! edn-states-atom nil)
        (.close ^java.io.OutputStream output-stream))
      (getDumpFileName [] file-path)
      (isNoop [] false)
      (isDot [] false)
      (isConstrained [] false)
      (snapshot [] nil))))

(defn- states-from-file
  [file-path]
  (with-open [is (io/input-stream file-path)]
    (let [reader (t/reader is :msgpack)]
      (t/read reader))))

(defn states-from-result
  [{:keys [:recife/transit-states-file-path]}]
  (states-from-file transit-states-file-path))

(defn random-traces-from-states
  ([states]
   (random-traces-from-states states {}))
  ([states {:keys [max-number-of-traces max-number-of-states]
            :or {max-number-of-traces 1}}]
   (if (< max-number-of-traces 1)
     []
     (let [removal-fn (if max-number-of-states
                        ;; If we have number of states defined, then
                        ;; we don't don't care about the visited states.
                        (fn [{:keys [paths]}]
                          (>= (count (last paths))
                              max-number-of-states))
                        (fn [{:keys [visited successors]}]
                          (contains? visited (first successors))))
           initial-states (get-in states [:ranks 0])]
       (loop [[current-state] [(rand-nth (vec initial-states)) nil]
              ;; `visited` is used to avoid loops.
              visited #{}
              ;; TODO: It can be improved to check for visited state successor.
              ;; Currently it does not return all possible paths (I wonder if
              ;; this is something we want), but it's fine for now.
              removed #{}
              paths [[[current-state nil]]]
              counter 0]
         (let [visited' (conj visited current-state)
               successor-state (some->> (get-in states [:states current-state :successors])
                                        (remove #(removal-fn {:visited visited
                                                              :successors %
                                                              :paths paths}))
                                        vec
                                        seq
                                        rand-nth)
               ;; If there is no successor state and the current path is a prefix
               ;; of some existing path, we can get rid of this path.
               paths' (if (and (nil? successor-state)
                               (some #(= (take (count (last paths)) %)
                                         (last paths))
                                     (drop-last paths)))
                        (vec (drop-last paths))
                        paths)
               ;; If all the successors are visited or removed, then we don't
               ;; need to check this state again.
               removed' (if (->> (get-in states [:states current-state :successors])
                                 (mapv first)
                                 (every? #(contains? (set/union removed visited') %)))
                          (conj removed current-state)
                          removed)]
           (cond
             (some? successor-state)
             (recur successor-state
                    visited'
                    removed'
                    (update paths' (dec (count paths')) (comp vec conj) successor-state)
                    (inc counter))

             ;; Stop if there are no more paths to see or if we have the desired
             ;; number of paths.
             (or (= (count paths') max-number-of-traces)
                 (= (count removed') (count (:states states)))
                 (not (->> (vec initial-states) (remove #(contains? removed' %)) seq)))
             ;; Return maps instead of positional data.
             ;; We make the output the same form as `:trace` when we have some
             ;; violation.
             (->> paths'
                  (mapv #(->> %
                              (map-indexed (fn [idx [state-fp context]]
                                             [idx (-> (get-in states [:states state-fp :state])
                                                      (merge (when (some? context)
                                                               {:recife/metadata {:context context}}))
                                                      (with-meta {:fingerprint state-fp}))]))
                              vec)))

             :else
             ;; Start a new path.
             (let [state (->> (vec initial-states)
                              (remove #(contains? removed' %))
                              rand-nth)]
               (recur [state nil]
                      #{}
                      removed'
                      (assoc paths' (count paths') [[state nil]])
                      (inc counter))))))))))

(defn random-traces-from-result
  ([result]
   (random-traces-from-result result {}))
  ([result opts]
   (random-traces-from-states (states-from-result result) opts)))

(defn- make-recorder
  "Create a recorder for capturing TLC messages."
  []
  (let [recorder-atom (atom {:others []})
        record #(swap! recorder-atom merge %)]
    {:atom recorder-atom
     :recorder (reify IMessagePrinterRecorder
                 (record [_ code objects]
                   (p* ::tlc-result--recorder
                       (condp = code
                         EC/TLC_INVARIANT_VIOLATED_BEHAVIOR
                         (record {:violation (merge
                                              {:type :invariant
                                               :name (to-edn-string (str/replace (first objects) #"_COLON_" ""))}
                                              (when (.exists (io/as-file invariant-data-filename))
                                                {:data (deserialize-obj invariant-data-filename)}))})

                         EC/TLC_ACTION_PROPERTY_VIOLATED_BEHAVIOR
                         (record {:violation {:type :action-property
                                              :name (to-edn-string (str/replace (first objects) #"_COLON_" ""))}})

                         EC/TLC_MODULE_VALUE_JAVA_METHOD_OVERRIDE
                         (record {:error-messages (str/split (last objects) #"\n")})

                         EC/TLC_INVARIANT_VIOLATED_INITIAL
                         (record {:violation (merge
                                              {:type :invariant
                                               :name (to-edn-string (str/replace (first objects) #"_COLON_" ""))
                                               :initial-state? true}
                                              (when (.exists (io/as-file invariant-data-filename))
                                                {:data (deserialize-obj invariant-data-filename)}))})

                         EC/TLC_DEADLOCK_REACHED
                         (record {:violation {:type :deadlock}})

                         EC/TLC_PARSING_FAILED
                         (record {:error :parsing-failure})

                         EC/GENERAL
                         (record {:error :general
                                  :error-messages (str/split (first objects) #"\n")})

                         (swap! recorder-atom update :others conj
                                [code objects])))))}))

(defn- build-result-from-tlc
  "Extract the result map from a TLC run."
  [^tlc2.TLC tlc recorder-info]
  (let [simulator tlc2.TLCGlobals/simulator
        {:keys [trace info]}
        (if-some [error-trace (-> ^tlc2.output.ErrorTraceMessagePrinterRecorder
                               (private-field tlc "recorder")
                                  .getMCErrorTrace
                                  (.orElse nil))]
          (let [states (->> (.getStates ^tlc2.model.MCError error-trace)
                            (mapv bean))
                stuttering-state (some #(when (:stuttering %) %) states)
                back-to-state (some #(when (:backToState %) %) states)]
            {:trace (->> states
                         (remove :stuttering)
                         (remove :backToState)
                         (mapv (fn [state]
                                 (->> state
                                      :variables
                                      (some #(when (= (.getName ^tlc2.model.MCVariable %)
                                                      "main_var")
                                               (.getTLCValue ^tlc2.model.MCVariable %))))))
                         (mapv tla-edn/to-edn)
                         (map-indexed (fn [idx v] [idx v]))
                         (into []))
             :info (merge (dissoc recorder-info :others)
                          (cond
                            stuttering-state
                            {:violation {:type :stuttering
                                         :state-number (:stateNumber stuttering-state)}}

                            back-to-state
                            {:violation {:type :back-to-state
                                         :state-number (dec (:stateNumber back-to-state))}}))})
          (let [initial-state (when (-> recorder-info :violation :initial-state?)
                                (some-> (private-field tlc2.TLCGlobals/mainChecker
                                                       tlc2.tool.AbstractChecker
                                                       "errState")
                                        bean
                                        :vals
                                        (.get (UniqueString/uniqueStringOf "main_var"))
                                        tla-edn/to-edn))]
            (cond
              initial-state
              {:trace [[0 initial-state]]
               :info (dissoc recorder-info :others)}

              (:error recorder-info)
              {:trace :error
               :info (dissoc recorder-info :others)}

              :else
              {:trace :ok})))]
    (-> {:trace (cond
                  (and (nil? (some-> ^tlc2.tool.AbstractChecker
                              tlc2.TLCGlobals/mainChecker
                                     .theFPSet
                                     .size))
                       (nil? simulator))
                  :error

                  :else
                  trace)
         :trace-info info
         :distinct-states (some-> tlc2.TLCGlobals/mainChecker .theFPSet .size)
         :generated-states (some-> tlc2.TLCGlobals/mainChecker .getStatesGenerated)
         :seed (private-field tlc "seed")
         :fp (private-field tlc "fpIndex")}
        (merge (when simulator
                 {:simulation
                  {:states-count (long (private-field simulator "numOfGenStates"))
                   :traces-count (long (private-field simulator "numOfGenTraces"))}})))))

(defn tlc-result-handler-local
  "Simplified handler for run-local mode. No subprocess infrastructure needed."
  [tlc-runner]
  ;; Use MemStateQueue instead of DiskStateQueue to avoid System.exit calls
  ;; from TLC's StatePoolReader thread on cleanup errors.
  (System/setProperty "tlc2.tool.queue.IStateQueue" "MemStateQueue")
  (let [{:keys [atom recorder]} (make-recorder)]
    (try
      (let [tlc (do (MP/setRecorder recorder)
                    (tlc-runner))]
        (p* ::process
            (doto ^tlc2.TLC tlc
              .process))
        (build-result-from-tlc tlc @atom))
      (catch Exception ex
        (serialize-obj ex exception-filename)
        {:trace :error
         :trace-info (.getMessage ex)
         :distinct-states (some-> tlc2.TLCGlobals/mainChecker .theFPSet .size)
         :generated-states (some-> tlc2.TLCGlobals/mainChecker .getStatesGenerated)})
      (finally
        (MP/unsubscribeRecorder recorder)))))

(defn tlc-result-handler
  "This function is a implementation detail, you should not use it.
  It handles the TLC object and its result, generating the output we see when
  calling `run-model`."
  [tlc-runner]
  (let [{:keys [atom recorder]} (make-recorder)
        ;; Read opts file from JVM property.
        {::keys [id] :as opts}
        (some-> (System/getProperty "RECIFE_OPTS_FILE_PATH") slurp edn/read-string)]
    (try
      (let [tlc (do (MP/setRecorder recorder)
                    (tlc-runner))
            _ (when-let [channel-path (::channel-file-path opts)]
                #_(println "Creating channel path at" channel-path)
                (r.buf/start-client-loop!)
                (r.buf/set-buf (r.buf/buf-create {:file (io/file channel-path)
                                                  :truncate true})))
            state-writer (when (or (:dump-states opts)
                                   ;; If we want to show a trace example (if no
                                   ;; violation is found), then we have to
                                   ;; generate the states file.
                                   (:trace-example opts))
                           (let [file-path (.getAbsolutePath (File/createTempFile "transit-output" ".msgpack"))
                                 os (io/output-stream file-path)
                                 state-writer (make-file-state-writer (t/writer os :msgpack) os (atom {}) file-path)]
                             (.setStateWriter ^tlc2.TLC tlc state-writer)
                             state-writer))
            *closed-properly? (atom false)
            _ (.addShutdownHook (Runtime/getRuntime)
                                (Thread. ^Runnable
                                 (fn []
                                   (r.buf/flush!)
                                   (when state-writer
                                     (.close state-writer))
                                   (when-not @*closed-properly?
                                     (println "\n---- Closing child Recife process ----\n")
                                     (let [pstats @@u/pd]
                                       (when (:stats pstats)
                                         (println (str "\n\n" (tufte/format-pstats pstats)))))
                                     (prn
                                      (merge
                                       (-> {:unfinished? true}
                                           (medley/assoc-some :recife/transit-states-file-path
                                                              (some-> state-writer .getDumpFileName)))
                                       {:trace (when (:trace-example opts)
                                                 (some-> state-writer
                                                         .getDumpFileName
                                                         states-from-file
                                                         random-traces-from-states
                                                         rand-nth))}))))))
            _ (save-and-flush! ::status [id :running])
            _ (save-and-flush! ::debug :tlc-start)
            _ (do (p* ::process
                      (doto ^tlc2.TLC tlc
                        .process))
                  (save-and-flush! ::debug :tlc-done)
                  (r.buf/flush!)
                  (save-and-flush! ::debug :flushed)
                  (reset! *closed-properly? true)
                  (let [pstats @@u/pd]
                    (when (:stats pstats)
                      (println (str "\n\n" (tufte/format-pstats pstats))))))
            recorder-info @atom
            _ (save-and-flush! ::debug :after-recorder-atom)]

        (save-and-flush! ::debug :after-building-violation-info)

        (-> (build-result-from-tlc tlc recorder-info)
            (merge (when (and (= (:trace (build-result-from-tlc tlc recorder-info)) :ok) (:trace-example opts))
                     {:trace (some-> state-writer
                                     .getDumpFileName
                                     states-from-file
                                     random-traces-from-states
                                     rand-nth)
                      :trace-info {:trace-example true}}))
            (merge (when (:continue opts)
                     (select-keys opts [:continue])))
            (medley/assoc-some :recife/transit-states-file-path (some-> state-writer .getDumpFileName))))

      (catch Exception ex
        (serialize-obj ex exception-filename)
        {:trace :error
         :trace-info (.getMessage ex)
         :distinct-states (some-> tlc2.TLCGlobals/mainChecker .theFPSet .size)
         :generated-states (some-> tlc2.TLCGlobals/mainChecker .getStatesGenerated)})
      (finally
        (MP/unsubscribeRecorder recorder)
        (save-and-flush! ::debug :before-flushing-done)
        (save-and-flush! ::status [id :done])
        (save-and-flush! ::debug :returning-from-tlc-result-handler)))))

(defn tlc-result-printer-handler
  [tlc-runner]
  (try
    (prn (tlc-result-handler tlc-runner))
    (finally
      (save-and-flush! ::debug :exiting)
      (System/exit 0))))

;; We create a record just so we can use `simple-dispatch`.
(defrecord RecifeModel [id *state v]
  java.io.Closeable
  (close [_]
    (.close v))

  clojure.lang.IDeref
  (deref [_]
    @v)

  Object
  (toString [_]
    "#RecifeModel {}")

  proto/IRecifeModel
  (-model-state [_]
    {:status (or (if-let [status (last (->> (read-saved-data ::status)
                                            (filter (comp #{id} first))
                                            (mapv last)))]
                   (do (swap! *state assoc :status status)
                       status)
                   (get @*state :status))
                 :waiting)}))

(defmethod print-method RecifeModel
  [_o ^java.io.Writer w]
  (.write w "#RecifeModel {}"))

(defmethod pp/simple-dispatch RecifeModel
  [^RecifeModel _]
  (pr {:type `RecifeModel}))

(defonce ^:private *current-model-run (atom nil))

(defn halt!
  "Halt model run. If no arg is passed, it halts the existing or last run (if
  existing).

  Returns @model-run."
  ([]
   (halt! @*current-model-run))
  ([model-run]
   (some-> model-run .close)
   (when (some? model-run)
     @model-run)))

(defn get-result
  "Wait for model run result and return it. If no arg is passed, it returns
  the last model run."
  ([]
   (get-result @*current-model-run))
  ([model-run]
   (when (some? model-run)
     @model-run)))

(defn get-model
  "Get current model."
  []
  @*current-model-run)

(defn- run-model*
  "Run model. Model run is async by default, use `halt!` to interrupt it."
  ([init-state next-operator operators]
   (run-model* init-state next-operator operators {}))
  ([init-state next-operator operators {:keys [seed fp workers tlc-args
                                               raw-output run-local isolated debug
                                               complete-response?
                                               no-deadlock
                                               depth async simulate generate
                                               continue
                                               ;; Below opts are used in the child
                                               ;; process.
                                               trace-example dump-states]
                                        :as opts
                                        :or {workers :auto
                                             async true
                                             run-local true
                                             isolated false}}]
   ;; Do some validation.
   (some->> (m/explain schema/Operator next-operator)
            me/humanize
            (hash-map :error)
            (ex-info "Next operator is invalid")
            throw)
   (some->> (m/explain [:set schema/Operator] operators)
            me/humanize
            (hash-map :error)
            (ex-info "Some operator is invalid")
            throw)
   (when-let [simple-keywords (seq (remove qualified-keyword? (keys init-state)))]
     (throw (ex-info "For the initial state, all the keywords should be namespaced. Recife uses the convention in which namespaced keywords are global ones, while other keywords are process-local."
                     {:keywords-without-namespace simple-keywords})))

   (try
     (halt!)
     (catch Exception _))

   ;; Run model.
   (let [use-buffer (not run-local)
         opts (if continue
                (assoc opts :trace-example false)
                opts)
         file (doto (File/createTempFile "eita" ".tla") .delete) ; we are interested in the random name of the folder
         abs-path (-> file .getAbsolutePath (str/split #"\.") first (str "/spec.tla"))
         opts-file-path (-> file .getAbsolutePath (str/split #"\.") first (str "/opts.edn"))
         _ (io/make-parents abs-path)
         [file-name file-type] (-> abs-path (str/split #"/") last (str/split #"\."))
         all-operators operators
         module-contents (module-template
                          {:init init-state
                           :next next-operator
                           :operators all-operators
                           :spec-name file-name})
         _ (spit abs-path module-contents)
         ;; Start client loop - needed for save! calls in invariants even in run-local mode.
         ;; For run-local, use simplified in-memory loop. For subprocess, use file-based buffer.
         _ (if run-local
             (r.buf/start-client-loop-local!)
             (do
               (r.buf/reset-buf!)
               (r.buf/sync!) ; sync so we can make sure that the writer can write
               (reset! r.buf/*contents {})
               (r.buf/start-client-loop!)
               (r.buf/start-sync-loop!)))
         ;; Also put a file with opts in the same folder so we can read configuration
         ;; in the tlc-handler function.
         id (gensym)
         _ (spit opts-file-path (merge (dissoc opts ::components)
                                       (when use-buffer
                                         {::channel-file-path (str @r.buf/*channel-file)
                                          ::id id})))
         tlc-opts (->> (cond-> ["-noTE" "-nowarning"]
                         simulate (conj "-simulate")
                         generate (cond->
                                   true (conj "-generate")
                                   (:num generate) (conj (str "num=" (:num generate))))
                         depth (conj "-depth" depth)
                         seed (conj "-seed" seed)
                         fp (conj "-fp" fp)
                         continue (conj "-continue")
                         no-deadlock (conj "-deadlock")
                         workers (conj "-workers" (if (keyword? workers)
                                                    (name workers)
                                                    workers))
                         (seq tlc-args) (concat tlc-args))
                       (mapv str))
         ;; Load only the namespaces of the classes which are necessary
         ;; so we don't have interference from other namespaces.
         ;; It's also faster!
         loaded-classes (->> all-operators
                             (mapv (juxt :op-ns :op-nss))
                             flatten
                             (remove nil?)
                             vec)]
     (when debug
       (println (->> (str/split-lines module-contents)
                     (map-indexed (fn [idx line]
                                    (str (inc idx) " " line)))
                     (str/join "\n"))))
     ;; Delete any serialization file.
     (io/delete-file exception-filename true)
     (io/delete-file invariant-data-filename true)
     (cond
       ;; Isolated mode uses a fresh class loader for each run,
       ;; avoiding TLC module caching issues when switching between specs.
       (and run-local isolated)
       (let [raw-result (spec/run-spec-isolated abs-path
                                                (str file-name "." file-type)
                                                tlc-opts
                                                {:loaded-classes loaded-classes})
             result (parse-tlc-isolated-output raw-result)]
         (if (.exists (io/as-file exception-filename))
           (throw (deserialize-obj exception-filename))
           result))

       run-local
       (let [result (tlc-result-handler-local #(spec/run-spec abs-path
                                                              (str file-name "." file-type)
                                                              tlc-opts
                                                              {:run? false}))]
         (if (.exists (io/as-file exception-filename))
           (throw (deserialize-obj exception-filename))
           ;; Add temporal properties check for consistency with subprocess mode
           (cond-> result
             (= (-> result :trace-info :violation :type) :back-to-state)
             (assoc-in [:experimental :violated-temporal-properties]
                       (rt/check-temporal-properties result (::components opts))))))

       raw-output
       (let [result (spec/run abs-path
                              (str file-name "." file-type)
                              tlc-opts
                              {:tlc-result-handler #'tlc-result-printer-handler
                               :complete-response? complete-response?
                               :loaded-classes loaded-classes})]
         (if (.exists (io/as-file exception-filename))
           (throw (deserialize-obj exception-filename))
           result))

       :else
       (let [result (spec/run abs-path
                              (str file-name "." file-type)
                              tlc-opts
                              {:tlc-result-handler #'tlc-result-printer-handler
                               :loaded-classes loaded-classes
                               :complete-response? true
                               :raw-args [(str "-DRECIFE_OPTS_FILE_PATH=" opts-file-path)]})
             output (atom [])
             t0 (System/nanoTime)
             *destroyed? (atom false)
             *streaming-finished? (atom false)
             -debug (fn [& v]
                      #_(apply println v)
                      nil)
             -deref-result (memoize
                            (fn []
                              ;; Wait until the process finishes.
                              (-debug :>>>__at-result)
                              @result
                              (-debug :>>>__after-at-result)
                              (reset! *destroyed? true)
                              (when use-buffer
                                (r.buf/stop-sync-loop!))
                              (-debug :>>>__after-stop-sync)
                              (while (not @*streaming-finished?))
                              (-debug :>>>__after-streaming-finished)
                              ;; Throw exception or return EDN result.
                              (when (empty? @output)
                                (let [msg (format "It seems we are unable to run Recife, see if there are error or if your classpath is correct and that it includes all the needed files (including the ns `%s`)!"
                                                  *ns*)]
                                  (println (ex-info msg {}))
                                  (throw (ex-info msg {}))))
                              (if (.exists (io/as-file exception-filename))
                                (throw (deserialize-obj exception-filename))
                                (try
                                  (let [edn (-> @output last edn/read-string)]
                                    (if (map? edn)
                                      (cond-> edn
                                        (= (-> edn :trace-info :violation :type) :back-to-state)
                                        ;; Tell the user which temporal properties are violated,
                                        ;; this still needs lots of testing!
                                        (assoc-in [:experimental :violated-temporal-properties]
                                                  (rt/check-temporal-properties edn (::components opts)))

                                        continue
                                        (assoc :violations (read-saved-data :recife/violation))

                                        true
                                        (with-meta {:type ::RecifeResponse}))
                                      (println (-> @output last))))
                                  (catch Exception _
                                    (println (-> @output last)))))))
             lock (gensym)
             deref-result (memoize (fn []
                                     (locking lock
                                       (-deref-result))))
             _output-streaming (future
                                 (with-open [rdr (io/reader (:out result))]
                                   (binding [*in* rdr]
                                     (loop [line (read-line)]
                                       (when line
                                         (when-let [last-line (last @output)]
                                           (println last-line))
                                         (swap! output conj line)
                                         (recur (read-line))))))
                                 (-debug :>>>__before-finishing-stream)
                                 (reset! *streaming-finished? true)
                                 (deref-result))
             process (RecifeModel.
                      id (atom {})
                      (reify
                        java.io.Closeable
                        (close [_]
                          (when-not @*destroyed?
                            (reset! *destroyed? true)
                            #_(p/destroy result)
                            (p/sh (format "kill -15 %s" (.pid (:proc result))))
                            (when use-buffer
                              (r.buf/stop-sync-loop!))
                            #_@output-streaming
                            (println (format "\n\n------- Recife process destroyed after %s seconds ------\n\n"
                                             (Math/round (/ (- (System/nanoTime)
                                                               t0)
                                                            1E9))))))

                        clojure.lang.IDeref
                        (deref [_]
                          #_@output-streaming
                          (deref-result))))]
         ;; Print errors
         #_(future
             (with-open [rdr (io/reader (:err result))]
               (binding [*in* rdr]
                 (loop [line (read-line)]
                   (when line
                     (println :err line)
                     (recur (read-line)))))))
         (reset! *current-model-run process)
         ;; Read line by line so we can stream the output to the user.
         (if async
           process
           @process))))))

(defn timeline-diff
  [result-map]
  (update result-map :trace
          (fn [result]
            (if (vector? result)
              (->> result
                   (mapv last)
                   (partition 2 1)
                   (mapv #(ddiff/diff (first %) (second %)))
                   (mapv (fn [step]
                           (->> step
                                (filter (fn [[k v]]
                                          ;; If the key or the value contains a diff
                                          ;; instance, keep it.
                                          (or (instance? Mismatch k)
                                              (instance? Deletion k)
                                              (instance? Insertion k)
                                              (let [result-atom (atom false)]
                                                (walk/prewalk (fn [form]
                                                                (when (or (instance? Mismatch form)
                                                                          (instance? Deletion form)
                                                                          (instance? Insertion form))
                                                                  (reset! result-atom true))
                                                                form)
                                                              v)
                                                @result-atom))))
                                (into {}))))
                   (cons (last (first result)))
                   (map-indexed (fn [idx step] [idx step]))
                   vec)
              result))))

(defn print-timeline-diff
  [result-map]
  (if (vector? (:trace result-map))
    (-> result-map
        timeline-diff
        (update :trace ddiff/pretty-print))
    result-map))

(defn run-model
  "Check `opts` in `run-model*`."
  ([init-global-state components]
   (run-model init-global-state components {}))
  ([init-global-state components opts]
   (if (System/getProperty "RECIFE_OPTS_FILE_PATH")
     (delay nil)
     (do
       (schema/explain-humanized schema/RunModelComponents components "Invalid components")
       (let [components (set (flatten (seq components)))
             processes (filter #(= (type %) RecifeProc) components)
             invariants (filter #(= (type %) ::Invariant) components)
             constraints (filter #(= (type %) ::Constraint) components)
             action-constraints (filter #(= (type %) ::ActionConstraint) components)
             properties (filter #(= (type %) ::Property) components)
             action-properties (filter #(= (type %) ::ActionProperty) components)
             fairness (filter #(= (type %) ::Fairness) components)
             procs (->> processes
                        (mapv :procs)
                        (apply merge))
             db-init (merge init-global-state
                            (when (seq procs)
                              {::procs procs}))
             next' (tla ::next
                        (->> processes
                             (mapv (fn [{:keys [:steps-keys :procs]}]
                                     [:exists {'self (set (keys procs))}
                                      (into [:or]
                                            (->> steps-keys
                                                 sort
                                                 (mapv #(vector % 'self))))]))
                             (cons :or)
                             vec))
             operators (set (concat (mapcat :operators processes)
                                    (map :operator invariants)
                                    (map :operator constraints)
                                    (map :operator action-constraints)
                                    (map :operator properties)
                                    (map :operator action-properties)
                                    (map :operator fairness)))]
         (run-model* db-init next' operators (assoc opts ::components components)))))))

(defmacro defproc
  "Defines a process and its multiple instances (`:procs`).

  `name` is a symbol for this var.

  `params` is optional a map of:
     `:procs` - set of arbitrary idenfiers (keywords);
     `:local` - map with local variables, `:pc` is required.

  `:pc` - initial step from the `steps` key (keyword).

  `steps` is a map of keywords (or vector) to functions. Each of
  these functions receives one argument (`db`) which contains the
  global state (namespaced keywords) plus any local state (unamespaced
  keywords) merged. E.g. in the example below, the function of `::check-funds`
  will have `:amount` available in it's input (besides all global state, which
  it uses only `:account/alice`).

  The keys in `steps` also can be a vector with two elements, the first one
  is the step identifier (keyword) and the second is some non-deterministic
  source (Recife model checker checks all the possibilities), e.g. if we want
  to model a failure, we can have `[::read {:notify-failure? #{true false}}]`
  instead of only `::read`, `:notify-failure` will be assoc'ed to into `db`
  of the associated function.


;; Example
(r/defproc wire {:procs #{:x :y}
                 :local {:amount (r/one-of (range 5))
                         :pc ::check-funds}}
  {::check-funds
   (fn [{:keys [:amount :account/alice] :as db}]
     (if (< amount alice)
       (r/goto db ::withdraw)
       (r/done db)))

   ::withdraw
   (fn [db]
     (-> db
         (update :account/alice - (:amount db))
         (r/goto ::deposit)))

   ::deposit
   (fn [db]
     (-> db
         (update :account/bob + (:amount db))
         r/done))})"
  {:arglists '([name params? steps])}
  [name & [params' steps']]
  (let [params (if (some? steps')
                 params'
                 {})
        steps (if (some? steps')
                steps'
                params')]
    `(def ~name
       (let [keywordized-name# (keyword (str *ns*) ~(str name))
             temp-steps# ~steps
             ;; If you pass a function to `steps`, it means that you
             ;; have only one step and the name of this will be derived
             ;; from `name`.
             steps# (if (fn? temp-steps#)
                      {keywordized-name# temp-steps#}
                      temp-steps#)
             temp-params# ~params
             ;; If we don't have `:procs`, use the name of the symbol as a proc name.
             procs# (or (:procs temp-params#)
                        #{keywordized-name#})
             ;; If we don't have `:local`, just use the first step (it should have
             ;; one only).
             local-variables# (or (:local temp-params#)
                                  {:pc (if (vector? (key (first steps#)))
                                         (first (key (first steps#)))
                                         (key (first steps#)))})
             params# {:procs procs#
                      :local local-variables#}]
         (schema/explain-humanized schema/DefProc ['~name params# steps#] "Invalid `defproc` args")
         (recife.records/map->RecifeProc
          {:name keywordized-name#
           :hash ~(hash [name params steps])
           :steps-keys (->> steps#
                            keys
                            (mapv #(if (vector? %)
                                     ;; If it's a vector, we are only interested
                                     ;; in the identifier.
                                     (first %)
                                     %)))
           :procs (->> (:procs params#)
                       (mapv (fn [proc#]
                               [proc# (:local params#)]))
                       (into {}))
           :operators (->> steps#
                           (mapv #(let [[k# opts#] (if (vector? (key %))
                                                     [(first (key %))
                                                      (or (second (key %)) {})]
                                                     [(key %)
                                                      {}])]
                                    (reg k#
                                         (with-meta opts#
                                           ~(meta name))
                                         (val %)))))})))))

(defmacro -definvariant
  [name & opts]
  `(def ~name
     (let [name# (keyword (str *ns*) ~(str name))
           [doc-string# f#] ~(if (= (count opts) 1)
                               [nil (first opts)]
                               [(first opts) (last opts)])]
       ^{:type ::Invariant}
       {:name name#
        :invariant f#
        :hash ~(hash [name opts])
        :operator (invariant name# doc-string# f#)})))

(defmacro -defproperty
  [name expr]
  `(def ~name
     (let [name# (keyword (str *ns*) ~(str name))
           expr# ~expr]
       ^{:type ::Property}
       {:name name#
        :property expr#
        :hash ~(hash [name expr])
        :operator (temporal-property name# expr#)})))

(defmacro -defaction-property
  [name expr]
  `(def ~name
     (let [name# (keyword (str *ns*) ~(str name))
           expr# ~expr]
       ^{:type ::ActionProperty}
       {:name name#
        :property expr#
        :hash ~(hash [name expr])
        :operator (temporal-property name# expr# {:action? true})})))

(defmacro -deffairness
  [name expr]
  `(def ~name
     (let [name# (keyword (str *ns*) ~(str name))
           expr# ~expr]
       ^{:type ::Fairness}
       {:name name#
        :property expr#
        :hash ~(hash [name expr])
        :operator (fairness name# expr#)})))

(defmacro -defconstraint
  [name f]
  `(def ~name
     (let [name# (keyword (str *ns*) ~(str name))
           f# ~f]
       ^{:type ::Constraint}
       {:name name#
        :constraint f#
        :hash ~(hash [name f])
        :operator (state-constraint name# f#)})))

(defmacro -defaction-constraint
  [name f]
  `(def ~name
     (let [name# (keyword (str *ns*) ~(str name))
           f# ~f]
       ^{:type ::ActionConstraint}
       {:name name#
        :constraint f#
        :hash ~(hash [name f])
        :operator (action-constraint name# f#)})))

(comment

  ;; TODO (from 2022-12-17):
  ;; - [x] Use Clerk for for visualization
  ;; - [x] Add action property support
  ;;   - [x] Operator
  ;;   - [x] Trace info
  ;; - [-] Add `next*` to rh
  ;;   - It can't be used in temporal properties (AFAIK), and action properties
  ;;     already receive the next state
  ;; - [x] Add documentation using Clerk
  ;; - [ ] Use it for real cases at work
  ;; - [ ] Ability to  query for provenance
  ;;   - [ ] E.g. how could I have an entity with such and such characteristics?
  ;;   - This would help us to ask questions about the system
  ;;   - [ ] Add doc
  ;;   - [-] Use datalog?
  ;;     - I guess not
  ;; - [ ] How to improve observability over what's being tested?
  ;;   - [ ] Gather real-time info about the running spec
  ;;   - [ ] Ability to check traces while running
  ;;     - [ ] Show it with clerk
  ;; - [ ] Create a def that can abstract a state machine
  ;; - [x] Add `defchecker` (complement of `definvariant`)
  ;; - [ ] Add a way to use locks without having the user having to reinvent the
  ;;       wheel every time
  ;; - [x] Add `not*` to rh
  ;; - [x] Add a way to tell which temporal properties were violated
  ;;   - https://github.com/tlaplus/tlaplus/issues/641
  ;; - [ ] Check a trace using properties and invariants without running the
  ;;       model
  ;;   - [ ] With it, we can create visualizations
  ;; - [ ] Can we build a spec without using the TLA+ compiler?
  ;;   - For perf purposes so we can avoid as many conversions as possible
  ;;   - [ ] Maybe we can leverage TLCGet to retrieve the initial values?
  ;; - [ ] Create var which holds state info
  ;; - [ ] Add a way to override default Spec expression
  ;; - [ ] Integrate with Soufflé (https://souffle-lang.github.io/source)?
  ;; - [ ] Warn about invalid temporal property?
  ;; - [ ] Add `["-lncheck" "final"]` by default
  ;; - [ ] Check if it's possible to do refinement
  ;;   - https://hillelwayne.com/post/refinement/#fnref:scope
  ;; - [ ] Visualize Recife state machine

  ())

(comment

  ;; We want to answer the question "Is RX programming using re-frame good
  ;; to model systems?"

  ;; We want it to be dynamic where you are able to create processes in
  ;; "runtime".

  ;; Everything is reified so you can manipulate anything that
  ;; it's hidden under the Recife functions. All the functions are just
  ;; normal Clojure code.

  ;; It's just an convention, but use namespaced keywords for "global" variables
  ;; and unamespaced keywords for "local" ones.

  ;; To mutate local variable, maybe you have to use some Recife function,
  ;; so the self is still hidden from you.

  ;; We can make it very flexible in the beginning, but we can constraint
  ;; later so it less overhead for simple models.

  ;; Different of TLA+/Pluscal, Recife requires you to explicitily define
  ;; sources of nondeterminism using `::r/or` or `::r/some` (TBD: `implies`
  ;; and `:in` should also be considered, see page 255, section 14.2.6
  ;; of https://lamport.azurewebsites.net/tla/book-02-08-08.pdf).

  ;; Another thing that you can do is to use the metadata (in `:recife/metadata`
  ;; for each step) to invoke your function with the same arguments as they were
  ;; invoked, allowing you to recreate all the steps. Just use the previous state
  ;; with a `[:recife/metadata :context]` step value.

  ;; TODO:
  ;; - Check if the arguments of `::r/or` could be simple functions (could
  ;;   naming be confusing for these cases?).
  ;; - Make (de)mangling of namespaces keywords (from)to tla values.
  ;; - One idea to make local variables work is just to override the
  ;;   implementattion of a map for `db`, see
  ;;   https://blog.wsscode.com/guide-to-custom-map-types/.
  ;; - [x] Make tla-edn accept keyword for a hashmap value (just use `str`).
  ;; - [x] Load dependencies correctly... `recife.core` does not want to load.

  ;; Recife TODO:
  ;; - [x] `Init` expression.
  ;; - [x] `reg` expression.
  ;; - [x] `Next` expression.
  ;; - [ ] Solve dependency problem (maybe by using vars instead of keywords?).
  ;; - [x] Define range for variables (one-of).
  ;; - [x] Add `init` properly.
  ;; - [x] Return output in EDN format (possibly with enhanced information about
  ;;       processes).
  ;; - [x] Temporal property.
  ;; - [ ] Add helper macros.
  ;; - [x] Encode back to state, stuttering, invariant which failed.
  ;; - [ ] Visualization of processes.
  ;; - [x] Check a way to avoid the error `Error: In evaluation, the identifier
  ;;       main_var is either undefined or not an operator.` when trying to
  ;;       override a operator called from a temporal property.
  ;; - [ ] It would be beneficial if side effects (e.g. `defop`) would be done
  ;;       when calling `r/run-model`.
  ;; - [x] Fix `Attempted to construct a set with too many elements (>1000000)."`.
  ;; - [ ] Profile parsing, things are slow in comparison with usual TLA+.
  ;; - [ ] (?) For performance reasons, make `db` be a custom map type.
  ;; - [x] Need to pass a seed so examples give us the same output.
  ;; - [x] Keep track of which process is acting so we know who is doing what.
  ;; - [x] Show `fp`, `seed`.
  ;; - [ ] Create source map from TLA+ to clojure expression?
  ;; - [ ] Print logs to stdout.
  ;; - [ ] Warn about invalid `::r/procs` (e.g. no `:pc`, (use malli)).
  ;; - [ ] Add functions to add/remove/use/map processes.
  ;; - [ ] Make init value reference each other (see `ch5-cache4`). Maybe make
  ;;       it able to receive a function which creates a anonymous
  ;;       operator override?
  ;; - [x] Add fairness.
  ;; - [ ] Probably don't need identifier for `next`, remove it.
  ;; - [ ] Don't make the user use `::r`, make it work with non-namespaced
  ;;       keywords.
  ;; - [ ] Maybe create some `:dispatch-n` where the mutations happen
  ;;       sequentially so you don't need to be too much verbose.
  ;; - [ ] Maybe add some extension point for parallelization, e.g. divide
  ;;       initial states and trigger `n` aws lambda executions. Maybe it's not
  ;;       worth it as for real specs the majority of states is not from initial?
  ;; - [ ] Maybe start a JVM preemptively so startup time is amortized? It would require
  ;;       some smartness regarding state of the code and when to load things,
  ;;       maybe it's not worth it.
  ;; - [ ] Maybe start TLC in the same JVM by reloading its classes.
  ;; - [ ] Fix `Error: Found a Java class for module spec, but unable to read"`
  ;;       error.
  ;; - [ ] Maybe give names to anonymous functions in `defproc`?
  ;; - [ ] Check coverage (e.g. if some operator is never enabled or if it's
  ;;       never enabled in some context).
  ;; - [ ] Create CHANGELOG file.
  ;; - [ ] Visualize states file.
  ;; - [ ] Convert `nil` values in Clojure to `:recife/null` in TLA+ and
  ;;       vice-versa.

  ;; PRIORITIES:
  ;; - [x] Better show which process caused which changes, it's still confusing.
  ;; - [x] Create TypeOkInvariant with `malli`.
  ;; - [x] Stream log so user doesn't stay in the dark.
  ;; - [x] For `ch6-threads-1` and `ch5-cache-3`, make it more convenient to
  ;;       create and input operators into `run-model`.
  ;; - [x] Make everyone use new `run-model`.
  ;; - [x] Maybe use `Elle` (https://github.com/jepsen-io/elle) to check for powerful
  ;;       invariants (linearizability, serializability etc). It shows that you
  ;;       can combine different libraries to augment your model checking.
  ;; - [x] `::/procs` could be created from some DSL.
  ;; - [x] Add some form to visualize the generated trace.
  ;; - [x] Throw exception if the params of `r/defproc` are invalid. Also check
  ;;       that initial `:pc` corresponds to one of the declared steps.
  ;; - [x] When running the model, check if there are any globally repeated
  ;;       `proc`s.
  ;; - [x] Add Hillel license.
  ;; - [x] When using `Elle`, some java process get focus from OSX,
  ;;       this is annoying.
  ;; - [x] Allow user to pass a function which will be used as a `CONSTRAINT`.
  ;; - [x] Maybe let the user pass custom messages from the invariant to the
  ;;       output.
  ;; - [ ] Show action which took it to a back to state violation.
  ;; - [ ] Use DFS and `arrudeia` to check implementations. Well, it appears
  ;;       that TLC DFS is not reliable, so we will have to use BFS, maybe
  ;;       generating all the possible traces upfront (with a `StateWriter`
  ;;       instance) and using it. Or just use BFS and require lots of memory
  ;;       for the user, it may be viable given the small scope hypothesis.
  ;; - [ ] Return `:error` in `:trace`, check for some kind of error code.
  ;; - [x] Remove bogus `-DTLCCustomHandler` JVM property.
  ;; - [ ] Make `r/run-model` ignore other `r/run-model` calls by checking a JVM
  ;;       property.
  ;; - [ ] Better documentation for fairness and `:exists` local variables.
  ;; - [ ] Create serialized files in a unique folder so it does not clash
  ;;       with concurrent runs of Recife.
  ;; - [ ] Add flag to return an trace example if no violation is found.
  ;; - [ ] Check that all initial global variables are namespaced.
  ;; - [ ] Use Pathom3, I don't want to be bothered how to get data (e.g.
  ;;       trace from the result map).
  ;; - [x] Fix a problem with the usage of empty maps.
  ;; - [ ] Return better error when some bogus thing happens in the user-provided
  ;;       functions.
  ;; - [ ] Add `tap` support.

  ())
